package ru.yandex.antifraud.data;

import java.net.InetAddress;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Currency;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.StringJoiner;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.currency.CurrencyMap;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.function.GenericFunction;
import ru.yandex.function.GenericUnaryOperator;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.util.ip.HbfParser;

public enum Field {
    ID("id"),
    EXTERNAL_ID("txn_extid", "extid", "ext_id", "external_id"),
    CHANNEL("channel"),
    CHANNEL_URI("channel_uri"),
    SUB_CHANNEL("sub_channel", "subchannel"),
    T("txn_timestamp", "t"),
    STATUS_TIMESTAMP("txn_status_timestamp", "t"),
    UID("uid", "src_client_id"),
    CARD_ID("card_id", "src_id"),
    CARD_BIN("card_bin", "src_parent"),
    CURRENCY("currency"),
    ORDER("order"),
    AMOUNT("amount", "amnt"),
    RUB_AMOUNT("rub_amount"),
    CARD_BRAND("card_brand"),
    IP("ip"),
    CARD_ISOA2("card_isoa2"),
    ORDER_TYPE("order_type"),
    PAYED_BY("payed_by"),
    GOOGLE_TOKEN_TYPE("google_token_type"),
    TAXI_CAR_NUMBER("taxi_car_number"),
    USER_PHONE("user_phone"),
    USER_AGENT("user_agent"),
    DEVICE_ID("device_id", "device_hardware_id", "uuid", "user_auth_id"),
    YID("yid", "yandexuid", "yandex_uid"),
    PAYMENT_METHOD("payment_method"),
    TERMINAL_ID("terminal_id"),
    REQUEST("request"),
    PAYMENT_TYPE("payment_type"),
    PAYMENT_MODE("payment_mode"),
    AS("as"),
    CARD_COUNTRY("card_isoa2"),
    TAXI_DRIVER_LICENSE("taxi_driver_license"),
    ORDER_ID("order_id"),
    ORDER_TARIF("order_tarif"),
    ORDER_TIPS("order_tips"),
    SUM_TO_PAY("sum_to_pay"),
    TRAFFIC_FP("traffic_fp"),
    DST_CLIENT_ID("dst_client_id"),
    CARD("card", "user_account"),
    ORDER_REGION_NAME("order_region_name"),
    MAIL("mail", "email"),
    HBF_ID("hbf_id", "ip"),
    SERVICE_PAYMENT_TYPE("service_payment_type"),
    VERIFICATION_LEVEL("verification_level"),
    STATUS("txn_status", "status"),
    CODE("txn_code", "code"),
    AUTHED("txn_is_authed", "is_authed"),
    COMMENT("txn_comment", "comment"),
    ACTION("txn_action"),
    TX_ID("txn_tx_id", "tx_id"),
    LOGIN_ID("login_id"),
    LOGIN_ID_SHORT("login_id_short", "login_id"),
    SERVICE("service_id"),
    AFS_VERIFICATION_LEVEL("afs_verification_level", "challenge", "phone_confirmation_method"),
    TRANSACTION_TYPE("transaction_type"),
    RESOLUTION("txn_afs_action"),
    APPLE_PAY_PAYED_BY("applepay"),
    GOOGLE_PAY_PAYED_BY("googlepay"),
    IS_SECURED("is_secured", "is_secure_phone");

    public static final List<Field> FIELDS_FOR_COUNTERS = Arrays.asList(
            Field.USER_PHONE,
            Field.IP,
            Field.AS,
            Field.USER_AGENT,
            Field.CARD_ID,
            Field.UID,
            Field.DEVICE_ID
    );

    public static final List<Field> FIELDS_FOR_PAYMENTS_LOG = Arrays.asList(
            Field.CARD_ID,
            Field.UID,
            Field.YID,
            Field.AMOUNT,
            Field.CURRENCY
    );

    private static final String PASSPORT_CARD_ID_PREFIX = "card-x";
    private static final String FAMILY_CARD_ID_PREFIX = "x";


    @Nonnull
    public static final String GET_ALL = "*";
    @Nonnull
    final String name;
    @Nonnull
    final List<String> aliases;
    final boolean isRequired;
    private final Value emptyValue;
    private final Multi singleMulti;

    Field(@Nonnull String name, boolean isRequired, @Nonnull List<String> aliases) {
        this.name = name;
        this.aliases = aliases;
        this.isRequired = isRequired;
        emptyValue = new Value(this);
        singleMulti = new Multi(this);
    }

    Field(@Nonnull String name, boolean isRequired) {
        this(name, isRequired, Collections.emptyList());
    }

    Field(@Nonnull String name) {
        this(name, false);
    }

    @SafeVarargs
    <T extends String> Field(@Nonnull String name, boolean isRequired, @Nonnull T... aliases) {
        this(name, isRequired, Arrays.asList(aliases));
    }

    @SafeVarargs
    <T extends String> Field(@Nonnull String name, @Nonnull T... aliases) {
        this(name, false, aliases);
    }

    @Nonnull
    public static Field parse(@Nonnull String src) {
        return Field.valueOf(src.trim());
    }

    @Nonnull
    public static String makeGetParam(@Nonnull List<Field> fields) {
        return fields.stream()
                .map(Field::fieldName)
                .collect(Collectors.joining(",")) + "," +
                "type," +
                "transaction_type," +
                "txn_afs_action," +
                "txn_afs_tags," +
                "txn_afs_reason," +
                "txn_afs_queue," +
                "txn_status," +
                "txn_is_authed," +
                "txn_status_timestamp," +
                "user_context";
    }

    @Nonnull
    public String fieldName() {
        return name;
    }

    @Nonnull
    public Value emptyValue() {
        return emptyValue;
    }

    @Nonnull
    public Multi singleMulti() {
        return singleMulti;
    }

    @Nonnull
    public List<String> filterBullshit(@Nullable JsonObject src) throws JsonBadCastException {
        if (src == null) {
            return Collections.emptyList();
        }
        switch (src.type()) {
            case BOOLEAN:
            case LONG:
            case DOUBLE:
            case STRING: {
                final String filtered = filterBullshit(src.asString());
                if (filtered != null) {
                    return Collections.singletonList(filtered);
                }
                break;
            }
            case LIST: {
                final List<String> list = new ArrayList<>(src.asList().size());
                for (JsonObject v : src.asList()) {
                    final String filtered = filterBullshit(v.asString());
                    if (filtered != null) {
                        list.add(filtered);
                    }
                }
                return list;
            }
        }
        return Collections.emptyList();
    }

    private static final Pattern NOT_DIGIT = Pattern.compile("[^\\d]");
    public static final Pattern BULLSHIT = Pattern.compile("\\s+|#|unknown|none|0|null", Pattern.CASE_INSENSITIVE);

    @Nullable
    public String filterBullshit(@Nullable String src) {
        if (src != null && !src.isEmpty() && !BULLSHIT.matcher(src).matches()) {
            switch (this) {
                case HBF_ID:
                case IP: {
                    final InetAddress ip = RblData.filterHost(src);
                    if (ip == null) {
                        return null;
                    }
                    final Long hbfId = HbfParser.INSTANCE.parseProjectId(ip);
                    if (hbfId != null) {
                        return this == IP ? null : Long.toHexString(hbfId);
                    } else {
                        return this == HBF_ID ? null : ip.getHostAddress();
                    }
                }
                case CURRENCY: {
                    final Currency currency = CurrencyMap.INSTANCE.getItemByStringCode(src);
                    if (currency != null) {
                        return currency.getCurrencyCode();
                    } else {
                        return src;
                    }
                }
                case USER_PHONE: {
                    return NOT_DIGIT.matcher(src).replaceAll("");
                }
                case LOGIN_ID_SHORT: {
                    final int colon = src.indexOf(':');

                    if (colon == -1) {
                        return src;
                    }

                    final int secondColon = src.indexOf(':', colon + 1);

                    if (secondColon != -1) {
                        return src.substring(0, secondColon);
                    } else {
                        return src;
                    }
                }
                case CARD_ID: {
                    return src.startsWith(PASSPORT_CARD_ID_PREFIX)
                            ? src.substring(PASSPORT_CARD_ID_PREFIX.length())
                            : src.startsWith(FAMILY_CARD_ID_PREFIX)
                            ? src.substring(FAMILY_CARD_ID_PREFIX.length())
                            : src;
                }
                default: {
                    return src;
                }
            }
        }

        return null;
    }

    @Nullable
    private JsonObject extractNotNull(@Nonnull JsonMap storage) {
        JsonObject result = storage.get(name);
        if (result == null || result == JsonNull.INSTANCE) {
            for (String field : aliases) {
                result = storage.get(field);
                if (result != null && result != JsonNull.INSTANCE) {
                    break;
                }
            }
        }
        return result;
    }

    @SafeVarargs
    @Nonnull
    public final <T extends JsonObject> List<String> extract(@Nonnull T... storages) throws JsonException {
        List<String> result = Collections.emptyList();
        for (JsonObject storage : storages) {
            if (storage != null && storage.type() == JsonObject.Type.MAP) {
                result = filterBullshit(extractNotNull(storage.asMap()));
                if (!result.isEmpty()) {
                    break;
                }
            }
        }

        if (result.isEmpty()) {
            switch (this) {
                case EXTERNAL_ID: {
                    result = Collections.singletonList("gen-" + UUID.randomUUID());
                    break;
                }
                case STATUS_TIMESTAMP:
                case T: {
                    result = Collections.singletonList(Long.toString(Instant.now().toEpochMilli()));
                }
            }
        }

        if (result.isEmpty()) {
            if (isRequired) {
                throw new JsonException("cannot find required field " + name + ", aliases:(" + String.join(",",
                        aliases) + ")");
            } else {
                return Collections.emptyList();
            }
        }

        return result;
    }

    public enum Parser
            implements GenericFunction<String, Field, Exception> {
        INSTANCE;

        @Override
        public Field apply(final String value) {
            return Field.valueOf(value.trim());
        }
    }

    public enum Validator
            implements GenericUnaryOperator<Field, Exception> {
        INSTANCE;

        @Override
        public Field apply(final Field value) {
            return value;
        }
    }

    public static class Multi {
        public static final Multi EMPTY = new Multi();

        private final Set<Field> fields;
        private final String fieldName;

        public Multi(@Nonnull Field... fields) {
            this(varToHashSet(fields));
        }

        public Multi(@Nonnull Set<Field> fields) {
            this.fields = fields;
            {
                final StringJoiner sj = new StringJoiner("_");
                for (Field field : fields) {
                    sj.add(field.fieldName());
                }
                this.fieldName = sj.toString();
            }
        }

        @Nonnull
        public Collection<Field> fields() {
            return fields;
        }

        @Override
        @Nonnull
        public String toString() {
            return fieldName;
        }

        public boolean contains(@Nullable Field field) {
            return fields.contains(field);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof Multi)) {
                return false;
            }

            final Multi other = (Multi) o;
            return Objects.equals(fieldName, other.fieldName);
        }

        @Override
        public int hashCode() {
            return Objects.hash(fieldName);
        }

        @Nonnull
        private static Set<Field> varToHashSet(@Nonnull Field[] fields) {
            switch (fields.length) {
                case 0:
                    return Collections.emptySet();
                case 1:
                    return Collections.singleton(fields[0]);
                default:
                    return new LinkedHashSet<>(Arrays.asList(fields));
            }
        }
    }
}
