package ru.yandex.antifraud.data;


import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class ListItem {
    public static final String ID = "id";
    public static final String LIST_NAME = "card";
    public static final String CHANNEL = "channel";
    public static final String SUB_CHANNEL = "sub_channel";
    public static final String CHANNEL_URI = "channel_uri";
    public static final String FROM = "txn_timestamp";
    public static final String TO = "txn_status_timestamp";
    public static final String AUTHOR = "uid";
    public static final String FIELD = "key";
    public static final String VALUE = "value";
    public static final String REASON = "txn_afs_reason";
    public static final String FIELD_VALUE = "key_value";
    public static final String TYPE = "type";

    @Nonnull
    private final String listName;
    @Nullable
    private final Field field;
    @Nonnull
    private final String value;
    @Nonnull
    private final TimeRange timeRange;
    @Nonnull
    private final String channel;
    @Nullable
    private final String subChannel;
    @Nullable
    private final String author;
    @Nullable
    private final String reason;

    public ListItem(@Nonnull String listName,
                    @Nullable Field field,
                    @Nonnull String value,
                    @Nonnull TimeRange timeRange,
                    @Nonnull ImmutableChannelConfig channelConfig,
                    @Nullable String author,
                    @Nonnull String reason) {
        this.listName = listName;
        this.field = field;
        this.value = value;
        this.timeRange = timeRange;
        this.channel = channelConfig.channel();
        this.subChannel = channelConfig.subChannel();
        this.author = author;
        this.reason = reason;
    }

    public ListItem(@Nonnull JsonMap src) throws JsonException,
            TimeRange.IllegalTimeRangeException {
        try {
            channel = src.getString(CHANNEL);
            subChannel = src.getString(SUB_CHANNEL, null);
            {
                final Instant from = Instant.ofEpochMilli(src.getLong(FROM));
                final Instant to = Instant.ofEpochMilli(src.getLong(TO));
                timeRange = new TimeRange(from, to);
            }
            {
                String listName = src.getString(LIST_NAME, null);
                if (listName == null) {
                    listName = src.getString("data");
                }
                this.listName = listName;
            }
            author = src.getString(AUTHOR);

            Field field;
            try {
                field = Optional.ofNullable(src.getString(FIELD, null)).map(f -> Field.valueOf(f.toUpperCase(Locale.ROOT))).orElse(null);
            } catch (Exception ignored) {
                field = null;
            }

            this.field = field;
            value = src.getString(VALUE);
            reason = src.getString(REASON, null);
        } catch (JsonException e) {
            throw new JsonException("cannot parse list item from " + JsonType.NORMAL.toString(src), e);
        }
    }

    @Nonnull
    public static String makeId(
            @Nonnull String channelUri,
            @Nonnull String listName,
            @Nonnull String value) {

        return "txn_" +
                channelUri +
                '_' +
                listName +
                '_' +
                value;
    }

    @Nonnull
    public static String makeId(
            @Nonnull ImmutableChannelConfig channelConfig,
            @Nonnull String listName,
            @Nonnull String value) {

        return makeId(channelConfig.channelUri(), listName, value);
    }

    @Nonnull
    public static List<ListItem> parseResponse(@Nonnull JsonObject response) throws JsonException,
            TimeRange.IllegalTimeRangeException {
        final JsonList rawItems = response.asMap().getList("hitsArray");
        final List<ListItem> lists = new ArrayList<>(rawItems.size());
        for (JsonObject rawItem : rawItems) {
            lists.add(new ListItem(rawItem.asMap()));
        }
        return lists;
    }

    public PrettyView asPretty() {
        return new PrettyView();
    }

    @Nonnull
    public String listName() {
        return listName;
    }

    @Nonnull
    public String channel() {
        return channel;
    }

    @Nullable
    public String subChannel() {
        return subChannel;
    }

    @Nonnull
    public String value() {
        return value;
    }

    @Nonnull
    public TimeRange timeRange() {
        return timeRange;
    }

    public class PrettyView implements JsonValue {
        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();

            writer.key("channel");
            writer.value(channel);

            if (subChannel != null) {
                writer.key("sub_channel");
                writer.value(subChannel);
            }

            writer.key("from");
            writer.value(timeRange.getStart().toEpochMilli());

            writer.key("to");
            writer.value(timeRange.getEnd().toEpochMilli());

            writer.key("list_name");
            writer.value(listName);

            writer.key("author");
            writer.value(author);

            if (field != null) {
                writer.key("field");
                writer.value(field.fieldName());
            }

            writer.key("value");
            writer.value(value);

            writer.key("reason");
            writer.value(reason);

            writer.endObject();
        }
    }
}
