package ru.yandex.antifraud.data;

import java.io.IOException;
import java.time.Instant;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;

import ru.yandex.antifraud.currency.Amount;
import ru.yandex.antifraud.currency.ToRubConverter;
import ru.yandex.antifraud.invariants.AfsVerificationLevel;
import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.antifraud.invariants.TransactionStatus;
import ru.yandex.antifraud.invariants.TransactionType;
import ru.yandex.antifraud.util.JsonWriterSb;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.lua.util.JsonUtils;

public class ScoringData {
    @Nullable
    protected final String channel;
    @Nullable
    protected final String subChannel;
    @Nonnull
    protected final String externalId;
    @Nonnull
    protected final Instant timestamp;
    @Nonnull
    private final JsonMap source;
    @Nonnull
    private final JsonMap truncatedSource;
    @Nullable
    private final TransactionType transactionType;
    @Nullable
    private final TransactionStatus transactionStatus;
    @Nullable
    private final ResolutionCode resolutionCode;
    @Nonnull
    private final EnumMap<Field, Value> values;
    @Nonnull
    private final Map<Field.Multi, Value.Multi> valuesCache = new HashMap<>();
    @Nonnull
    private final LuaTable normalized;

    public ScoringData(@Nonnull JsonMap source) throws JsonException {
        this(source, null);
    }

    public ScoringData(@Nonnull JsonMap source, @Nullable ToRubConverter toRubConverter) throws JsonException {
        this.values = new EnumMap<>(Field.class);

        final JsonMap afsParams = source.getMapOrNull("afs_params");
        final JsonMap afsParameters = source.getMapOrNull("afs_parameters");
        for (Field field : Field.values()) {
            final List<String> valuesList = field.extract(
                    afsParams,
                    afsParameters,
                    source
            );
            if (!valuesList.isEmpty()) {
                values.put(field, new Value(field, valuesList));
            }
        }

        this.source = source;
        this.truncatedSource = JsonUtils.truncate(source, 5);

        channel = values.getOrDefault(Field.CHANNEL, Field.CHANNEL.emptyValue()).value();
        subChannel = values.getOrDefault(Field.SUB_CHANNEL, Field.SUB_CHANNEL.emptyValue()).value();
        timestamp = Instant.ofEpochMilli(values.getOrDefault(Field.T, Field.T.emptyValue()).longValue());

        {
            final String rawTransactionType = values.getOrDefault(Field.TRANSACTION_TYPE,
                    Field.TRANSACTION_TYPE.emptyValue()).value();

            if (rawTransactionType != null) {
                transactionType = TransactionType.valueOf(rawTransactionType);
            } else {
                transactionType = TransactionType.parseFromPaymentType(
                        getValue(Field.PAYMENT_TYPE).value(),
                        getValue(Field.PAYMENT_MODE).value(),
                        getValue(Field.REQUEST).value()
                );
                if (transactionType != null) {
                    values.put(Field.TRANSACTION_TYPE, new Value(Field.TRANSACTION_TYPE, transactionType.toString()));
                }
            }
        }

        {
            final Value transactionStatus = values.getOrDefault(Field.STATUS, null);
            if (transactionStatus != null) {
                this.transactionStatus = TransactionStatus.valueOf(transactionStatus.nonNullValue());
            } else {
                this.transactionStatus = null;
            }
        }

        {
            final Value resolutionCode = values.getOrDefault(Field.RESOLUTION, null);
            if (resolutionCode != null) {
                this.resolutionCode = ResolutionCode.valueOf(resolutionCode.nonNullValue());
            } else {
                this.resolutionCode = null;
            }
        }

        if (!values.containsKey(Field.TX_ID)) {
            values.put(Field.TX_ID, new Value(Field.TX_ID, makeTxId()));
        }

        if (!values.containsKey(Field.RUB_AMOUNT)) {
            final Value amount = values.getOrDefault(Field.AMOUNT, null);
            if (amount != null) {
                final Value currency = values.getOrDefault(Field.CURRENCY, null);
                final Long rubAmount;
                if (toRubConverter != null && currency != null) {
                    rubAmount = toRubConverter.apply(new Amount(amount.longValue(), currency.nonNullValue()));
                } else {
                    rubAmount = amount.longValue();
                }
                values.put(Field.RUB_AMOUNT, new Value(Field.RUB_AMOUNT, rubAmount.toString()));
            }
        }

        if (!values.containsKey(Field.AFS_VERIFICATION_LEVEL)) {
            @Nullable final String afsVerificationLevel = AfsVerificationLevel.calcAfsVerificationLevel(
                    getValue(Field.PAYMENT_TYPE).value(),
                    source.getMapOrNull("verification_details"),
                    getValue(Field.AUTHED).LongValue());

            if (afsVerificationLevel != null) {
                values.put(Field.AFS_VERIFICATION_LEVEL, new Value(
                        Field.AFS_VERIFICATION_LEVEL,
                        afsVerificationLevel));
            }
        }

        final String externalId = values.get(Field.EXTERNAL_ID).nonNullValue();
        if (transactionType == TransactionType.PRE_CHECK) {
            this.externalId = externalId + "_pre_check";
            values.put(Field.EXTERNAL_ID, new Value(
                    Field.EXTERNAL_ID,
                    this.externalId));
        } else {
            this.externalId = externalId;
        }

        {
            final Value uid = values.getOrDefault(Field.UID, null);
            final Value payedBy = values.getOrDefault(Field.PAYED_BY, null);
            if (uid != null && payedBy != null) {
                if ("googlepay".equals(payedBy.nonNullValue())) {
                    values.put(Field.GOOGLE_PAY_PAYED_BY, new Value(Field.GOOGLE_PAY_PAYED_BY, "googlepay"));
                }
                if ("applepay".equals(payedBy.nonNullValue())) {
                    values.put(Field.APPLE_PAY_PAYED_BY, new Value(Field.APPLE_PAY_PAYED_BY, "applepay"));
                }
            }
        }

        normalized = makeNormalized();
    }

    @Nullable
    public String getChannel() {
        return channel;
    }

    @Nullable
    public String getSubChannel() {
        return subChannel;
    }

    @Nonnull
    public String getExternalId() {
        return externalId;
    }

    @Nonnull
    public Instant getTimestamp() {
        return timestamp;
    }

    public String shortInfo() {
        return externalId + ' ' +
                channel + ' ' +
                subChannel;
    }

    @Nonnull
    public String makeTxId() {
        return channel + '/' + subChannel + ':' + externalId;
    }


    @Nonnull
    public Value getValue(@Nonnull Field field) {
        return values.getOrDefault(field, field.emptyValue());
    }

    @Nonnull
    public Value.Multi getMultiValue(@Nonnull Field.Multi fields) {
        return valuesCache.computeIfAbsent(fields,
                ignored -> new Value.Multi(this::getValue, fields));
    }

    @Nullable
    public String getGoogleTokenType() {
        return getValue(Field.GOOGLE_TOKEN_TYPE).value();
    }

    @Nonnull
    public JsonMap asJson() {
        return source;
    }

    @Nonnull
    public JsonMap asTruncatedJson() {
        return truncatedSource;
    }

    @Nonnull
    private LuaTable makeNormalized() {
        final LuaTable normalized = new LuaTable();
        for (Value value : values.values()) {
            if (!value.isEmpty()) {
                normalized.set(value.field().fieldName(), value.value());
            }
        }

        {
            final Value rubAmount = values.getOrDefault(Field.RUB_AMOUNT, null);
            if (rubAmount != null) {
                normalized.set(Field.RUB_AMOUNT.fieldName(), rubAmount.longValue());
            }
        }
        {
            final Value amount = values.getOrDefault(Field.AMOUNT, null);
            if (amount != null) {
                normalized.set(Field.AMOUNT.fieldName(), amount.longValue());
            }
        }

        return normalized;
    }

    @Nonnull
    public LuaTable normalized() {
        return normalized;
    }

    @Nullable
    public String getOrderId() {
        return getValue(Field.ORDER_ID).value();
    }

    @Nullable
    public Long getCardBin() {
        return getValue(Field.CARD_BIN).LongValue();
    }

    public long getAmount() {
        return getValue(Field.AMOUNT).longValue();
    }

    public long getRubAmount() {
        return getValue(Field.RUB_AMOUNT).longValue();
    }

    @Nullable
    public String getCurrency() {
        return getValue(Field.CURRENCY).value();
    }

    @Nullable
    public String getIp() {
        return getValue(Field.IP).value();
    }

    @Nullable
    public String getHbfId() {
        return getValue(Field.HBF_ID).value();
    }

    @Nullable
    public String getServiceId() {
        return getValue(Field.SERVICE).value();
    }

    @Nullable
    public String getIpOrHbfId() {
        final String ip = getIp();
        if (ip != null) {
            return ip;
        }
        return getHbfId();
    }


    @Nullable
    public String getOrderTarif() {
        return getValue(Field.ORDER_TARIF).value();
    }

    public long getOrderTips() {
        return getValue(Field.ORDER_TIPS).longValue();
    }

    public long getSumToPay() {
        return getValue(Field.SUM_TO_PAY).longValue();
    }

    @Nullable
    public String getTaxiDriverLicense() {
        return getValue(Field.TAXI_DRIVER_LICENSE).value();
    }

    @Nullable
    public String getTaxiCarNumber() {
        return getValue(Field.TAXI_CAR_NUMBER).value();
    }

    @Nullable
    public String getCardId() {
        return getValue(Field.CARD_ID).value();
    }

    @Nullable
    public String getUid() {
        return getValue(Field.UID).value();
    }

    @Nullable
    public String getRegionName() {
        return getValue(Field.ORDER_REGION_NAME).value();
    }

    @Nullable
    public String getIsoCountry() {
        return getValue(Field.CARD_COUNTRY).value();
    }

    @Nullable
    public String getMail() {
        return getValue(Field.MAIL).value();
    }

    @Nullable
    public String getRequest() {
        return getValue(Field.REQUEST).value();
    }

    @Nullable
    public TransactionType getTransactionType() {
        return transactionType;
    }

    @Nullable
    public String getPayedBy() {
        return getValue(Field.PAYED_BY).value();
    }

    @Nullable
    public String getDeviceId() {
        return getValue(Field.DEVICE_ID).value();
    }

    @Nullable
    public String getPhone() {
        return getValue(Field.USER_PHONE).value();
    }

    @Nonnull
    public List<String> getAsList() {
        return getValue(Field.AS).values();
    }

    @Nullable
    public String getYid() {
        return getValue(Field.YID).value();
    }

    @Nonnull
    public JsonWriterSb makeBaseLogData(@Nonnull Set<String> fieldsToExclude) throws IOException {
        final JsonWriterSb logData = JsonWriterSb.create();

        logData.key("id");
        logData.value(externalId);

        logData.key("transaction_type");
        logData.value(transactionType);

        logData.key("channel");
        logData.value(channel);

        logData.key("subchannel");
        logData.value(subChannel);

        logData.key("unixtime");
        logData.value(Instant.now().getEpochSecond());

        if (getHbfId() != null) {
            logData.key("hbf_id");
            logData.value(getHbfId());
        }

        logData.key("request_data");
        logData.startObject();
        for (Map.Entry<String, JsonObject> entry : source.entrySet()) {
            if (!fieldsToExclude.contains(entry.getKey())) {
                logData.key(entry.getKey());
                logData.value(entry.getValue());
            }
        }
        logData.endObject();

        logData.key("nsrc");
        logData.value(new JsonUtils.LuaAsJson(normalized()));

        for (LuaValue key : normalized.keys()) {
            logData.key("nsrc." + key.tojstring());
            logData.value(new JsonUtils.LuaAsJson(normalized.get(key)));
        }


        return logData;
    }

    @Nullable
    public String getPaymentMethod() {
        return getValue(Field.PAYMENT_METHOD).value();
    }

    @Nullable
    public TransactionStatus getTransactionStatus() {
        return transactionStatus;
    }

    @Nullable
    public ResolutionCode getResolutionCode() {
        return resolutionCode;
    }

    @Nullable
    public Long isAuthed() {
        return getValue(Field.AUTHED).LongValue();
    }
}

