package ru.yandex.antifraud.data;

import java.io.IOException;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.antifraud.invariants.TransactionStatus;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class StorageData implements JsonValue {
    @Nonnull
    private final ScoringData request;
    @Nonnull
    private final List<String> queues;
    @Nonnull
    private final JsonMap sourceData;

    public StorageData(@Nonnull JsonMap data) throws JsonException {
        sourceData = data;
        {
            final String rawRequestData = data.asMap().getString("data", null);

            if (rawRequestData != null) {
                request = new ScoringData(TypesafeValueContentHandler.parse(rawRequestData).asMap());
            } else {
                request = new ScoringData(data);
            }
        }
        {
            final String queuesStr = data.getString("txn_afs_queue", null);
            if (queuesStr != null) {
                queues = Arrays.asList(queuesStr.split("\\s+"));
            } else {
                queues = Collections.emptyList();
            }
        }
    }


    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        sourceData.put("data", request.asJson());
        writer.value((JsonValue) sourceData);
    }

    @Nonnull
    public ScoringData getRequest() {
        return request;
    }

    @Nullable
    public TransactionStatus getTransactionStatus() {
        return request.getTransactionStatus();
    }

    @Nullable
    public ResolutionCode getResolutionCode() {
        return request.getResolutionCode();
    }

    @Nullable
    public TransactionStatus getTransactionStatusAccordingToUpdateTimestamp(@Nonnull Instant now) {
        final TransactionStatus transactionStatus = request.getTransactionStatus();
        final Instant statusTimestamp = Instant.ofEpochMilli(request.getValue(Field.STATUS_TIMESTAMP).longValue());
        return statusTimestamp != null && !statusTimestamp.isBefore(now) ? null : transactionStatus;
    }

    @Nonnull
    public List<String> getQueues() {
        return queues;
    }

    @Nullable
    public Boolean isAuthed() {
        final Long isAuthed = request.getValue(Field.AUTHED).LongValue();
        return isAuthed != null ? isAuthed == 1 : null;
    }
}
