package ru.yandex.antifraud.data;

import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;

import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import fingerprint.v1.FingerprintV1;
import org.bouncycastle.util.encoders.Hex;

import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonDouble;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;

public enum TrafficFeaturesMaker {
    INSTANCE;

    private static final List<String> SYN_TCP_FEATURES = Arrays.asList(
            "syn_mss",
            "syn_opt_eol_pad",
            "syn_quirk_df",
            "syn_quirk_ecn",
            "syn_quirk_flow",
            "syn_quirk_nz_ack",
            "syn_quirk_nz_id",
            "syn_quirk_nz_mbz",
            "syn_quirk_nz_urg",
            "syn_quirk_opt_bad",
            "syn_quirk_opt_eol_nz",
            "syn_quirk_opt_exws",
            "syn_quirk_opt_nz_ts2",
            "syn_quirk_opt_zero_ts1",
            "syn_quirk_push",
            "syn_quirk_urg",
            "syn_quirk_zero_ack",
            "syn_quirk_zero_id",
            "syn_quirk_zero_seq",
            "syn_win",
            "syn_win_scale",
            "syn_win_type",
            "no_ts_opt_percent"
    );
    private static final List<String> SSL_CIPHERSUITES_FEATURES = Stream.of(
            "0001", "0002", "0003", "0004", "0005", "0006",
            "0007", "0008", "0009", "000a", "000c", "000d",
            "0010", "0011", "0012", "0013", "0014", "0015",
            "0016", "0018", "001a", "001b", "001c", "001e",
            "001f", "0020", "0022", "0023", "0024", "002f",
            "0030", "0031", "0032", "0033", "0034", "0035",
            "0036", "0037", "0038", "0039", "003a", "003b",
            "003c", "003d", "003e", "003f", "0040", "0041",
            "0042", "0043", "0044", "0045", "0060", "0067",
            "0068", "0069", "006a", "006b", "006c", "006d",
            "0081", "0084", "0085", "0086", "0087", "0088",
            "0094", "0096", "0097", "0098", "0099", "009a",
            "009c", "009d", "009e", "009f", "00a0", "00a1",
            "00a2", "00a3", "00a4", "00a5", "00a6", "00a7",
            "00ba", "00bd", "00be", "00c0", "00c3", "00c4",
            "00ff", "05ec", "0a0a", "1301", "1302", "1303",
            "1304", "1a1a", "201a", "268d", "2986", "2a2a",
            "3a3a", "4a4a", "5600", "5a5a", "5e3f", "6a6a",
            "7a40", "7a7a", "8221", "8a8a", "901d", "9a9a",
            "aaaa", "b0ca", "baba", "c001", "c002", "c003",
            "c004", "c005", "c006", "c007", "c008", "c009",
            "c00a", "c00b", "c00c", "c00d", "c00e", "c00f",
            "c010", "c011", "c012", "c013", "c014", "c015",
            "c016", "c017", "c018", "c019", "c01b", "c01c",
            "c01e", "c01f", "c021", "c022", "c023", "c024",
            "c025", "c026", "c027", "c028", "c029", "c02a",
            "c02b", "c02c", "c02d", "c02e", "c02f", "c030",
            "c031", "c032", "c050", "c051", "c052", "c053",
            "c056", "c057", "c05c", "c05d", "c060", "c061",
            "c072", "c073", "c076", "c077", "c09c", "c09d",
            "c09e", "c09f", "c0a0", "c0a1", "c0a2", "c0a3",
            "c0ac", "c0ad", "c0ae", "c0af", "caca", "cca8",
            "cca9", "ccaa", "d0c2", "dada", "eaea", "fafa",
            "ff85"
    ).map(suite -> "ssl_ciphersuites_" + suite).collect(Collectors.toList());
    private static final List<String> SSL_EXTENSIONS_FEATURES = Stream.of(
            "0000", "0001", "0005", "000a", "000b", "000d",
            "000f", "0010", "0011", "0012", "0015", "0016",
            "0017", "001b", "001c", "0023", "002b", "002d",
            "0031", "0032", "0033", "00ff", "03cb", "0a0a",
            "0c29", "0dc9", "1195", "1301", "1a1a", "1db4",
            "2a2a", "3374", "3a3a", "3a7c", "4a4a", "5a5a",
            "61cb", "69ab", "6a6a", "7550", "7a7a", "8a8a",
            "9317", "9a9a", "aaaa", "baba", "c99b", "caca",
            "d0c2", "dada", "eaea", "fafa", "ff01"
    ).map(suite -> "ssl_extensions_" + suite).collect(Collectors.toList());
    private static final List<String> SUSPICIOUS_SSL_FEATURES = Arrays.asList(
            "suspicious_ssl_client_version",
            "suspicious_ssl_extension_size",
            "suspicious_ssl_compression_size",
            "suspicious_ssl_record_size",
            "suspicious_ssl_all_extensions_size",
            "suspicious_ssl_handshake_size"
    );
    private static final List<String> CLIENT_VERSIONS = Arrays.asList("0303", "0301", "0302", "not_set", "0304");
    private static final List<String> CLIENT_VERSIONS_FEATURES = CLIENT_VERSIONS.stream()
            .map(version -> "ssl_client_version_" + version).collect(Collectors.toList());
    private static final List<String> PROTOCOL_VERSIONS_FEATURES = Stream
            .of("0301", "0303", "not_set", "0300", "0302")
            .map(version -> "ssl_protocol_version_" + version).collect(Collectors.toList());

    @Nonnull
    public JsonMap makeFeatures(@Nonnull String rawProto) throws InvalidProtocolBufferException {
        final byte[] decoded = Base64.getDecoder().decode(rawProto);
        final FingerprintV1.Fingerprint trafficFP = FingerprintV1.Fingerprint.parseFrom(decoded);
        return makeFeatures(trafficFP);
    }

    @Nonnull
    public JsonMap makeFeatures(@Nonnull FingerprintV1.Fingerprint trafficFP) {
        final JsonMap features = new JsonMap(BasicContainerFactory.INSTANCE);
        final FingerprintV1.BinaryParams binaryParams = trafficFP.getBinaryParams();

        for (List<String> featuresList : Arrays.asList(
                SYN_TCP_FEATURES,
                SSL_CIPHERSUITES_FEATURES,
                SSL_EXTENSIONS_FEATURES,
                SSL_EXTENSIONS_FEATURES,
                SUSPICIOUS_SSL_FEATURES,
                CLIENT_VERSIONS_FEATURES,
                PROTOCOL_VERSIONS_FEATURES
        )) {
            for (String feature : featuresList) {
                features.put(feature, new JsonDouble(-1.));
            }
        }

        if (binaryParams.getSynMss() != 0) {
            features.put("syn_mss", JsonLong.valueOf(binaryParams.getSynMss()));
        }
        if (binaryParams.getSynOptEolPad() != 0) {
            features.put("syn_opt_eol_pad", JsonLong.valueOf(binaryParams.getSynOptEolPad()));
        }
        if (binaryParams.getSynQuirkDf()) {
            features.put("syn_quirk_df", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkEcn()) {
            features.put("syn_quirk_ecn", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkFlow()) {
            features.put("syn_quirk_flow", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkNzAck()) {
            features.put("syn_quirk_nz_ack", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkNzId()) {
            features.put("syn_quirk_nz_id", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkNzMbz()) {
            features.put("syn_quirk_nz_mbz", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkNzUrg()) {
            features.put("syn_quirk_nz_urg", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkOptBad()) {
            features.put("syn_quirk_opt_bad", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkOptEolNz()) {
            features.put("syn_quirk_opt_eol_nz", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkOptExws()) {
            features.put("syn_quirk_opt_exws", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkOptNzTs2()) {
            features.put("syn_quirk_opt_nz_ts2", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkOptZeroTs1()) {
            features.put("syn_quirk_opt_zero_ts1", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkPush()) {
            features.put("syn_quirk_push", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkUrg()) {
            features.put("syn_quirk_urg", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkZeroAck()) {
            features.put("syn_quirk_zero_ack", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkZeroId()) {
            features.put("syn_quirk_zero_id", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynQuirkZeroSeq()) {
            features.put("syn_quirk_zero_seq", JsonLong.valueOf(1));
        }
        if (binaryParams.getSynWin() != 0) {
            features.put("syn_win", JsonLong.valueOf(binaryParams.getSynWin()));
        }
        if (binaryParams.getSynWinScale() != 0) {
            features.put("syn_win_scale", JsonLong.valueOf(binaryParams.getSynWinScale()));
        }
        if (binaryParams.getSynWinType() != 0) {
            features.put("syn_win_type", JsonLong.valueOf(binaryParams.getSynWinType()));
        }

        if (binaryParams.getNoTsOptPercent() > 0) {
            features.put("no_ts_opt_percent", new JsonDouble(binaryParams.getNoTsOptPercent()));
        }

        final FingerprintV1.TlsParams tlsParams = binaryParams.getTlsParams();

        for (int i = 0; i < tlsParams.getSslCiphersuitesCount(); i++) {
            final ByteString cipherSuite = tlsParams.getSslCiphersuites(i);
            final String featureName =
                    "ssl_ciphersuites_" + Hex.toHexString(cipherSuite.toByteArray());
            features.put(featureName, new JsonDouble(1.f / (1 + i)));
        }

        for (FingerprintV1.TlsExtension tlsExtension : tlsParams.getSslExtensionsList()) {
            final String featureName =
                    "ssl_extensions_" + Hex.toHexString(tlsExtension.getKey().toByteArray());
            features.put(featureName, JsonLong.valueOf(1));
        }

        if (tlsParams.getSuspiciousSslClientVersion()) {
            features.put("suspicious_ssl_client_version", JsonLong.valueOf(1));
        }
        if (tlsParams.getSuspiciousSslExtensionSize()) {
            features.put("suspicious_ssl_extension_size", JsonLong.valueOf(1));
        }
        if (tlsParams.getSuspiciousSslCompressionSize()) {
            features.put("suspicious_ssl_compression_size", JsonLong.valueOf(1));
        }
        if (tlsParams.getSuspiciousSslRecordSize()) {
            features.put("suspicious_ssl_record_size", JsonLong.valueOf(1));
        }
        if (tlsParams.getSuspiciousSslAllExtensionsSize()) {
            features.put("suspicious_ssl_all_extensions_size", JsonLong.valueOf(1));
        }
        if (tlsParams.getSuspiciousSslHandshakeSize()) {
            features.put("suspicious_ssl_handshake_size", JsonLong.valueOf(1));
        }

        if (tlsParams.getSslClientVersion() != 0) {
            for (String version : CLIENT_VERSIONS) {
                final String hexed = Integer.toHexString(tlsParams.getSslClientVersion());
                if (version.equals(hexed)) {
                    features.put("ssl_client_version_" + hexed, JsonLong.valueOf(1));
                }
            }
        } else {
            features.put("ssl_client_version_not_set", JsonLong.valueOf(1));
        }

        if (tlsParams.getSslProtocolVersion() != 0) {
            for (String version : CLIENT_VERSIONS) {
                final String hexed = Integer.toHexString(tlsParams.getSslProtocolVersion());
                if (version.equals(hexed)) {
                    features.put("ssl_protocol_version" + hexed, JsonLong.valueOf(1));
                }
            }
        } else {
            features.put("ssl_protocol_version_not_set", JsonLong.valueOf(1));
        }

        return features;
    }
}
