package ru.yandex.antifraud.data;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;

public class Value implements Comparable<Value> {
    @Nonnull
    private final Field field;
    @Nonnull
    private final List<String> value;

    public Value(@Nonnull Field field) {
        this.field = field;
        value = Collections.emptyList();
    }

    public Value(@Nonnull Field field, @Nonnull String value) {
        this(field, Collections.singletonList(value));
    }

    public Value(@Nonnull Field field, @Nonnull List<String> values) {
        this.field = field;
        this.value = values;
    }

    @Nonnull
    public static Value extract(@Nonnull Field field, @Nonnull JsonMap src) throws JsonException {
        final List<String> valuesList = field.extract(src);

        if (!valuesList.isEmpty()) {
            return new Value(field, valuesList);
        } else {
            return field.emptyValue();
        }
    }

    public static long longValue(@Nonnull List<String> value, long defaultValue) {
        return value.isEmpty() ? defaultValue : Long.parseLong(value.get(0));
    }

    @Nonnull
    public List<String> values() {
        return value;
    }

    @Nullable
    public String value() {
        return value((String) null);
    }

    @Nullable
    public String value(String defaultValue) {
        return value(() -> defaultValue);
    }

    @Nullable
    public String value(Supplier<String> valueSupplier) {
        return value.isEmpty() ? valueSupplier.get() : value.get(0);
    }

    @Nonnull
    public String nonNullValue() {
        if (value.isEmpty()) {
            throw new NullPointerException("field " + field + " is not set");
        }
        return value.get(0);
    }

    @Override
    public boolean equals(Object another) {
        if (another instanceof Value) {
            final Value anotherValue = (Value) another;
            return Objects.equals(field, anotherValue.field) && Objects.equals(value, anotherValue.value);
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        return field.hashCode() ^ value.hashCode();
    }

    @Override
    @Nonnull
    public String toString() {
        return field + ":" + value;
    }

    public long longValue() {
        return value.isEmpty() ? 0L : Long.parseLong(value.get(0));
    }

    public long longValue(long defaultValue) {
        return longValue(value, defaultValue);
    }

    @Nullable
    public Long LongValue() {
        return value.isEmpty() ? null : Long.parseLong(value.get(0));
    }

    @Nonnull
    public Field field() {
        return field;
    }

    public boolean isEmpty() {
        return value.isEmpty() || value.stream().anyMatch(Objects::isNull);
    }

    @Override
    public int compareTo(Value value) {
        return field.compareTo(value.field);
    }

    public static class Multi {
        @Nonnull
        public static final Multi EMPTY = new Multi();
        @Nonnull
        private final Field.Multi multiField;
        private final boolean empty;
        @Nullable
        private final String value;

        private final int hashCode;

        private Multi() {
            this.value = null;
            this.empty = true;
            this.hashCode = 0;
            this.multiField = Field.Multi.EMPTY;
        }

        public Multi(@Nonnull Function<Field, Value> valueProvider, @Nonnull Field.Multi fields) {
            {
                boolean empty = false;
                int hashCode = 0;
                final StringJoiner sj = new StringJoiner("_");
                for (Field field : fields.fields()) {
                    final Value value = valueProvider.apply(field);
                    if (value.isEmpty()) {
                        empty = true;
                        break;
                    }
                    final String v = value.nonNullValue();
                    sj.add(v);
                    hashCode ^= v.hashCode();
                }
                this.value = !empty ? sj.toString() : null;
                this.empty = empty;
                this.hashCode = hashCode;
            }
            this.multiField = fields;
        }

        @Nullable
        public String value() {
            return value;
        }

        @Nonnull
        public Field.Multi fields() {
            return multiField;
        }

        @Override
        public boolean equals(Object another) {
            if (another instanceof Multi) {
                final Multi anotherValue = (Multi) another;
                return this == anotherValue || Objects.equals(value, anotherValue.value);
            } else {
                return false;
            }
        }

        @Override
        public int hashCode() {
            return hashCode;
        }

        @Override
        @Nullable
        public String toString() {
            return multiField.toString() + '_' + value;
        }

        public boolean isEmpty() {
            return empty;
        }
    }
}
