package ru.yandex.antifraud.lua_context_manager;

import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.lib.OneArgFunction;
import core.org.luaj.vm2.lib.ZeroArgFunction;
import jse.org.luaj.vm2.lib.jse.CoerceJavaToLua;

import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.json.dom.JsonList;
import ru.yandex.lua.util.JsonUtils;

public enum ArtefactsTuner {
    INSTANCE;

    public void tuneLuaContext(@Nonnull LuaValue context, @Nonnull Artefacts artefacts) {
        context.set("ALLOW", new VoidFunction(unused -> artefacts.getResolution().allow()));
        context.set("DENY", new VoidFunction(unused -> artefacts.getResolution().deny()));
        context.set("set_status_code", new IntConsumerFunction(artefacts::statusCode));
        context.set("tag", new CollectJsonFunction(artefacts.getResolution().getTags()));
        context.set("comment", new StringConsumerFunction(s -> artefacts.getResolution().getReason().add(s)));
        context.set("reason", new StringConsumerFunction(s -> artefacts.getResolution().getReason().add(s)));
        context.set("custom", new LogTuner.KeyValueLogFunction(artefacts.getResolution().customFields()));

        context.set("addToQueue", new StringConsumerFunction(s -> artefacts.getQueues().add(s)));

        context.set("counters", CoerceJavaToLua.coerce(artefacts.getCounters()));

        context.set("client", CoerceJavaToLua.coerce(artefacts.luaClient()));

        context.set("interrupt", new VoidFunction(unused -> artefacts.interrupt()));

        context.set("resolution",
                new ValueFunction(() -> LuaValue.valueOf(artefacts.getResolution().resolutionCode().name())));

        context.set("tags",
                new ValueFunction(() -> JsonUtils.jsonToLua(artefacts.getResolution().getTags())));

        context.set("comments",
                new ValueFunction(() -> JsonUtils.toLua(artefacts.getResolution().getReason())));

    }

    private static class ValueFunction extends ZeroArgFunction {
        final Supplier<LuaValue> function;

        public ValueFunction(Supplier<LuaValue> function) {
            this.function = function;
        }

        @Override
        public LuaValue call() {
            return function.get();
        }
    }

    private static class VoidFunction extends ZeroArgFunction {
        final Consumer<Void> function;

        public VoidFunction(Consumer<Void> function) {
            this.function = function;
        }

        @Override
        public LuaValue call() {
            function.accept(null);
            return LuaValue.NIL;
        }
    }

    private static class StringConsumerFunction extends OneArgFunction {
        final private Consumer<String> consumer;

        public StringConsumerFunction(Consumer<String> consumer) {
            this.consumer = consumer;
        }

        @Override
        public LuaValue call(LuaValue x) {
            final String s = x.tojstring();
            if(s != null) {
                consumer.accept(s);
            }
            return LuaValue.NIL;
        }
    }

    private static class IntConsumerFunction extends OneArgFunction {
        final private Consumer<Integer> consumer;

        public IntConsumerFunction(Consumer<Integer> consumer) {
            this.consumer = consumer;
        }

        @Override
        public LuaValue call(LuaValue x) {
            consumer.accept(x.toint());
            return LuaValue.NIL;
        }
    }

    private static class CollectJsonFunction extends OneArgFunction {
        final private JsonList refToList;

        public CollectJsonFunction(JsonList l) {
            refToList = l;
        }

        @Override
        public LuaValue call(LuaValue x) {
            refToList.add(JsonUtils.luaToJson(x, refToList.containerFactory()));
            return LuaValue.NIL;
        }
    }
}
