package ru.yandex.antifraud.lua_context_manager;

import java.util.Currency;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaError;
import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.lib.OneArgFunction;
import core.org.luaj.vm2.lib.TwoArgFunction;

import ru.yandex.antifraud.currency.CurrencyMap;
import ru.yandex.antifraud.currency.CurrencyRateMap;

public class CurrenciesRatesTuner extends LuaPackage {
    @Nonnull
    private final CurrencyRateMap rates;

    public CurrenciesRatesTuner(@Nonnull CurrencyRateMap rates) {
        super("currency");
        this.rates = rates;
    }

    @Override
    public void setup(LuaTable pack) {
        pack.set("rate", new RateFunction());
        pack.set("to_rub_rate", new ToRubRateFunction());
    }

    class RateFunction extends TwoArgFunction {
        @Override
        public LuaValue call(LuaValue from, LuaValue to) {
            final Currency fromCurrency = CurrencyMap.INSTANCE.getItemByName(from.checkjstring());
            if (fromCurrency == null) {
                throw new LuaError("from currency is null");
            }

            final Currency toCurrency = CurrencyMap.INSTANCE.getItemByName(to.checkjstring());
            if (toCurrency == null) {
                throw new LuaError("to currency is null");
            }

            final Double rate = rates.rate(fromCurrency, toCurrency);
            if (rate == null) {
                throw new LuaError("cannot find rate from " + from + " to " + to);
            }

            return LuaValue.valueOf(rate);
        }
    }

    class ToRubRateFunction extends OneArgFunction {
        @Override
        public LuaValue call(LuaValue to) {
            final Currency toCurrency = CurrencyMap.INSTANCE.getItemByName(to.checkjstring());
            if (toCurrency == null) {
                throw new LuaError("to currency is null");
            }

            final Double rate = rates.rateToRub(toCurrency);
            if (rate == null) {
                throw new LuaError("cannot find rate from rub to " + to);
            }

            return LuaValue.valueOf(rate);
        }
    }
}
