package ru.yandex.antifraud.lua_context_manager;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ListItem;
import ru.yandex.antifraud.lua_context_manager.config.ImmutablePrototypesConfig;

public class FileListsProvider implements ListsProvider {
    @Nonnull
    private final ListChecker listChecker;

    public FileListsProvider(@Nonnull ImmutablePrototypesConfig prototypesConfig) throws IOException,
            InvalidFormatException {

        final List<ListItem> lists;

        if (!prototypesConfig.channels().isEmpty()) {
            lists = new ArrayList<>();
            for (ImmutableChannelConfig channelConfig : prototypesConfig.channels().values()) {

                final List<Path> dirs = new ArrayList<>();
                for (List<String> listsDirs : Arrays.asList(
                        channelConfig.listDir(),
                        prototypesConfig.additionalListsDir())) {
                    if (listsDirs == null) {
                        continue;
                    }
                    for (String dir : listsDirs) {
                        dirs.add(Path.of(dir));
                    }
                }

                for (final Path dir : dirs) {
                    try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir, "*.csv")) {
                        for (final Path p : stream) {
                            final String listName = getFileNameOnly(p.getFileName().toString());
                            try (final BufferedReader r = getFileReader(p)) {
                                try {
                                    for (Map.Entry<String, TimeRange> kv : readList(r).entrySet()) {
                                        final ListItem item = new ListItem(
                                                listName,
                                                null,
                                                kv.getKey(),
                                                kv.getValue(),
                                                channelConfig,
                                                null,
                                                "");
                                        lists.add(item);
                                    }
                                } catch (RuntimeException e) {
                                    throw new InvalidFormatException("error in file " + p, e);
                                }
                            }
                        }
                    }
                }

            }
        } else {
            lists = Collections.emptyList();
        }

        listChecker = new ListChecker(lists);
    }

    public static Map<String, TimeRange> readList(BufferedReader r) throws InvalidFormatException {
        final Map<String, TimeRange> res = new HashMap<>();
        // columns are: value, ... , createdat, expereat
        for (Iterator<String> it = r.lines().skip(1).iterator(); it.hasNext(); ) {
            final String row = it.next();
            if (row.isEmpty()) {
                continue;
            }

            final String[] parts = row.split("\\t");

            if (parts.length < 3) {
                throw new InvalidFormatException(row, "row must contains at" +
                        " least three parts: value\t(some " +
                        "comments\t)*start\tend");
            }

            try {
                final Instant from = parseUntilField(parts[parts.length - 2]);
                final Instant until = parseUntilField(parts[parts.length - 1]);
                res.put(parts[0], new TimeRange(from, until));
            } catch (NumberFormatException | DateTimeParseException | TimeRange.IllegalTimeRangeException e) {
                throw new InvalidFormatException(row, e);
            }
        }

        return res;
    }

    public static String getFileNameOnly(String s) {
        String[] row = s.split("\\.");
        if (row.length <= 2) {
            return row[0];
        }
        return s.substring(0, s.length() - (row[row.length - 1].length() + 1));
    }

    private static BufferedReader getFileReader(Path p) throws IOException {
        return new BufferedReader(new FileReader(p.toString(), Charset.defaultCharset()));
    }

    @Nonnull
    private static Instant parseUntilField(@Nonnull String untilField) throws NumberFormatException,
            DateTimeParseException {
        if (untilField.matches("\\d+")) {
            return Instant.ofEpochMilli(Long.parseLong(untilField));
        } else {
            return Instant.parse(untilField);
        }
    }

    @Nonnull
    @Override
    public ListChecker getLists() {
        return listChecker;
    }

    public static class InvalidFormatException extends Exception {
        static final long serialVersionUID = 0L;

        public InvalidFormatException(String line, String cause) {
            super("invalid format in line: '" + line + "' cause " + cause);
        }

        public InvalidFormatException(String line, Exception e) {
            super("invalid format in line: '" + line + "'", e);
        }

        public InvalidFormatException(String message,
                                      InvalidFormatException e) {
            super(message, e);
        }
    }
}
