package ru.yandex.antifraud.lua_context_manager;

import java.util.List;

import javax.annotation.Nonnull;

import com.google.i18n.phonenumbers.NumberParseException;
import com.google.i18n.phonenumbers.PhoneNumberUtil;
import com.google.i18n.phonenumbers.Phonemetadata;
import com.google.i18n.phonenumbers.Phonenumber;
import core.org.luaj.vm2.LuaError;
import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.Varargs;
import core.org.luaj.vm2.lib.OneArgFunction;
import core.org.luaj.vm2.lib.VarArgFunction;
import jse.org.luaj.vm2.lib.jse.CoerceJavaToLua;

public class LibPhoneNumberTuner extends LuaPackage {
    public static final LibPhoneNumberTuner INSTANCE = new LibPhoneNumberTuner();

    public LibPhoneNumberTuner() {
        super("libphonenumber");
    }

    @Override
    public void setup(LuaTable pack) {
        pack.set("e164", new FormatNumberFunction());
        pack.set("parse", new ParseFunction());
    }


    public static class PhoneNumberBinding {
        private static final List<Phonemetadata.NumberFormat> FORMATS = List.of(
                Phonemetadata.NumberFormat.newBuilder()
                        .setFormat(" ($1) $2-$3-$4")
                        .setPattern("(\\d{3})(\\d{3})(\\d{2})(\\d{2})"));
        @Nonnull
        private final Phonenumber.PhoneNumber parsed;

        public PhoneNumberBinding(@Nonnull String phone) throws LuaError {
            try {
                this.parsed = PhoneNumberUtil.getInstance().parse(phone, "RU");
            } catch (NumberParseException e) {
                throw new LuaError(e);
            }
        }

        @LuaBinding
        @Nonnull
        public String formatE164() {
            return PhoneNumberUtil.getInstance().format(parsed, PhoneNumberUtil.PhoneNumberFormat.E164);
        }

        @LuaBinding
        @Nonnull
        public String formatInternational() {
            return PhoneNumberUtil.getInstance().formatByPattern(
                    parsed,
                    PhoneNumberUtil.PhoneNumberFormat.E164,
                    FORMATS);
        }

        @LuaBinding
        @Nonnull
        public String regionCodeForNumber() {
            return PhoneNumberUtil.getInstance().getRegionCodeForNumber(parsed);
        }
    }

    static class FormatNumberFunction extends VarArgFunction {
        @Override
        public Varargs invoke(Varargs args) {
            final String phone = args.arg(1).tojstring();
            final PhoneNumberBinding parsed = new PhoneNumberBinding(phone);
            return LuaValue.varargsOf(
                    LuaValue.valueOf(parsed.formatE164()),
                    LuaValue.valueOf(parsed.formatInternational()));
        }
    }

    static class ParseFunction extends OneArgFunction {
        @Override
        public LuaValue call(LuaValue phone) {
            return CoerceJavaToLua.coerce(new PhoneNumberBinding(phone.tojstring()));
        }
    }
}
