package ru.yandex.antifraud.lua_context_manager;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaBoolean;
import core.org.luaj.vm2.LuaError;
import core.org.luaj.vm2.LuaNil;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.Varargs;
import core.org.luaj.vm2.lib.VarArgFunction;
import jse.org.luaj.vm2.lib.jse.CoerceJavaToLua;

import ru.yandex.antifraud.artefacts.PreparedLists;
import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.storage.ListUpdateRequest;
import ru.yandex.antifraud.storage.UpdateRequest;

public enum ListTuner {
    INSTANCE;

    private static final String IN_LIST = "inList";
    private static final String IN_LOCAL_LIST = "inLocalList";
    private static final String PUT_IN_LIST = "putInList";
    private static final String TO_CHECK_IN_LIST = "toCheckInList";

    public static void tuneLuaContext(@Nonnull ChannelResolver channelResolver,
                                      @Nonnull LuaValue context,
                                      @Nonnull Instant now,
                                      @Nonnull ImmutableChannelConfig channelConfig,
                                      @Nonnull List<ListsProvider> listsProviders,
                                      @Nonnull List<String> checkedLists,
                                      @Nonnull List<UpdateRequest> listsRequests) {

        final InListFunction inListFunction = new InListFunction(
                now,
                channelConfig,
                listsProviders.stream().map(ListsProvider::getLists).collect(Collectors.toList()),
                checkedLists,
                channelResolver);

        context.set(IN_LIST, inListFunction);
        context.set(IN_LOCAL_LIST, inListFunction);

        context.set(PUT_IN_LIST,
                new ListsUpdateFunction(
                        channelResolver,
                        now,
                        channelConfig,
                        listsRequests));
    }

    public static void tuneLuaContext(@Nonnull LuaValue context,
                                      @Nonnull Instant now,
                                      @Nonnull String channel,
                                      @Nullable String subChannel,
                                      @Nonnull PreparedLists preparedLists) {
        context.set(TO_CHECK_IN_LIST, new PreparedListsFunction(now, channel, subChannel, preparedLists));
    }

    private static class InListFunction extends VarArgFunction {
        @Nonnull
        private final Instant now;
        @Nonnull
        private final ImmutableChannelConfig channelConfig;
        @Nonnull
        private final List<ListsProvider.ListChecker> listCheckers;
        @Nonnull
        private final List<String> checkedLists;
        @Nonnull
        private final ChannelResolver channelResolver;

        InListFunction(@Nonnull Instant now,
                       @Nonnull ImmutableChannelConfig channelConfig,
                       @Nonnull List<ListsProvider.ListChecker> listCheckers,
                       @Nonnull List<String> checkedLists, @Nonnull ChannelResolver channelResolver) {
            this.now = now;
            this.channelConfig = channelConfig;
            this.listCheckers = listCheckers;
            this.checkedLists = checkedLists;
            this.channelResolver = channelResolver;
        }

        @Override
        public Varargs invoke(Varargs args) {
            @Nonnull final String channel;
            @Nullable final String subChannel;
            @Nonnull final String listName;
            @Nonnull final String value;

            switch (args.narg()) {
                case 2:
                    channel = channelConfig.channel();
                    subChannel = channelConfig.subChannel();
                    listName = args.arg(1).tojstring();
                    value = args.arg(2).tojstring();
                    break;
                case 3:
                    channel = args.arg(1).tojstring();
                    subChannel = null;
                    listName = args.arg(2).tojstring();
                    value = args.arg(3).tojstring();
                    break;
                case 4:
                    channel = args.arg(1).tojstring();
                    subChannel = args.arg(2).tojstring();
                    listName = args.arg(3).tojstring();
                    value = args.arg(4).tojstring();
                    break;
                default:
                    throw new LuaError("invalid args count :" + args.narg());
            }

            @Nonnull final ImmutableChannelConfig channelConfig;
            try {
                channelConfig =
                        channel.equals(this.channelConfig.channel()) && Objects.equals(subChannel,
                                this.channelConfig.subChannel())
                                ? this.channelConfig
                                : channelResolver.resolve(channel, subChannel);
            } catch (UnknownChannelException e) {
                throw new LuaError(e);
            }

            return LuaBoolean.valueOf(checkRange(channelConfig, listName, value));
        }

        public boolean checkRange(@Nonnull ImmutableChannelConfig channelConfig,
                                  @Nonnull String listName,
                                  @Nonnull String value) {
            for (ListsProvider.ListChecker listChecker : listCheckers) {
                final TimeRange range = listChecker.getRange(channelConfig, listName, value);
                if (range != null) {
                    if (range.contains(now)) {
                        checkedLists.add(listName + ":" + value + ":1");
                        return true;
                    } else {
                        checkedLists.add(listName + ":" + value + ":rotten");
                    }
                }
            }
            checkedLists.add(listName + ":" + value + ":0");
            return false;
        }
    }

    public static class ListsUpdateFunction extends VarArgFunction {
        @Nonnull
        private final ChannelResolver channelResolver;
        @Nonnull
        private final Instant now;
        @Nonnull
        private final ImmutableChannelConfig channelConfig;
        @Nonnull
        private final List<UpdateRequest> listsRequests;

        public ListsUpdateFunction(@Nonnull ChannelResolver channelResolver,
                                   @Nonnull Instant now,
                                   @Nonnull ImmutableChannelConfig channelConfig,
                                   @Nonnull List<UpdateRequest> listsRequests) {
            this.channelResolver = channelResolver;
            this.now = now;
            this.channelConfig = channelConfig;
            this.listsRequests = listsRequests;
        }

        @Override
        public Varargs invoke(Varargs args) {
            @Nullable final String channel;
            @Nullable final String subChannel;

            final int offset;
            switch (args.narg()) {
                case 5:
                    offset = 0;
                    channel = this.channelConfig.channel();
                    subChannel = this.channelConfig.subChannel();
                    break;
                case 6:
                    offset = 1;
                    channel = args.arg(1).checkjstring();
                    subChannel = null;
                    break;
                case 7:
                    offset = 2;
                    channel = args.arg(1).checkjstring();
                    subChannel = args.arg(2).checkjstring();
                    break;
                default:
                    throw new LuaError("wrong args number:" + args.narg() + ", must be 6,7 or 8 args");
            }

            @Nonnull final String listName = args.arg(offset + 1).checkjstring();
            @Nonnull final String value = args.arg(offset + 2).checkjstring();
            @Nonnull final String author = args.arg(offset + 3).checkjstring();
            @Nonnull final String reason = args.arg(offset + 4).checkjstring();
            final int days = args.arg(offset + 5).checkint();

            try {

                final ImmutableChannelConfig channelConfig = Objects.equals(channel, this.channelConfig.channel()) &&
                        Objects.equals(subChannel, this.channelConfig.subChannel())
                        ? this.channelConfig
                        : channelResolver.resolve(channel, subChannel);

                listsRequests.add(
                        new ListUpdateRequest(
                                channelConfig,
                                listName,
                                author,
                                reason,
                                Collections.singletonList(value),
                                new TimeRange(
                                        now,
                                        now.plus(Duration.ofDays(days))
                                ))
                );
            } catch (TimeRange.IllegalTimeRangeException | UnknownChannelException e) {
                throw new LuaError(e);
            }

            return LuaNil.NIL;
        }
    }

    public static class PreparedListsFunction extends VarArgFunction {
        @Nonnull
        private final Instant now;
        @Nonnull
        private final String channel;
        @Nullable
        private final String subChannel;
        @Nonnull
        private final PreparedLists preparedLists;

        public PreparedListsFunction(@Nonnull Instant now,
                                     @Nonnull String channel,
                                     @Nullable String subChannel,
                                     @Nonnull PreparedLists preparedLists) {
            this.now = now;
            this.channel = channel;
            this.subChannel = subChannel;
            this.preparedLists = preparedLists;
        }

        @Override
        public Varargs invoke(Varargs args) {
            @Nullable final String channel;
            @Nullable final String subChannel;

            final int offset;
            switch (args.narg()) {
                case 2:
                    offset = 0;
                    channel = this.channel;
                    subChannel = this.subChannel;
                    break;
                case 3:
                    offset = 1;
                    channel = args.arg(1).tojstring();
                    subChannel = null;
                    break;
                case 4:
                    offset = 2;
                    channel = args.arg(1).tojstring();
                    subChannel = args.arg(2).tojstring();
                    break;
                default:
                    throw new LuaError("invalid args count :" + args.narg());
            }

            @Nonnull final String listName = args.arg(offset + 1).tojstring();
            @Nonnull final LuaValue luaValue = args.arg(offset + 2);

            final String value;
            if (luaValue.isnumber()) {
                value = Long.toString(luaValue.tolong());
            } else {
                value = luaValue.tojstring();
            }

            final PreparedLists.ValueToCheck valueToCheck;
            try {
                valueToCheck = preparedLists.add(
                        now,
                        channel,
                        subChannel,
                        listName,
                        value);
            } catch (UnknownChannelException e) {
                throw new LuaError(e);
            }

            return CoerceJavaToLua.coerce(valueToCheck);
        }
    }

}
