package ru.yandex.antifraud.lua_context_manager;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.Channel;
import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.ChannelWithCrossChannels;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.lua_context_manager.config.ImmutablePrototypesConfig;
import ru.yandex.lua.util.LuaException;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.parser.config.ConfigException;

public class PrototypesManager implements ChannelResolver {
    private final Map<ImmutableChannelConfig, ChannelWithCrossChannels> channels = new HashMap<>();

    @Nonnull
    private final FileListsProvider listsProvider;

    @Nonnull
    private final ImmutablePrototypesConfig prototypesConfig;

    @Nonnull
    private final MetricRegistry metricRegistryRoot = new MetricRegistry();

    public PrototypesManager(@Nonnull ImmutablePrototypesConfig prototypesConfig,
                             @Nonnull Map<String, YasmTuner> yasmTuners,
                             @Nullable UaTraitsTuner uaTraitsTuner,
                             @Nullable CurrenciesRatesTuner currenciesRatesTuner) throws IOException,
            ConfigException, LuaException, FileListsProvider.InvalidFormatException {
        this.prototypesConfig = prototypesConfig;
        final Map<String, Channel> channelsByKey = new HashMap<>();

        listsProvider = new FileListsProvider(prototypesConfig);

        for (Map.Entry<String, ImmutableChannelConfig> kv : prototypesConfig.channels().entrySet()) {
            final String name = kv.getKey();
            final ImmutableChannelConfig config = kv.getValue();

            final Channel channel = new Channel(config,
                    prototypesConfig.additionalScriptsDir(),
                    yasmTuners.computeIfAbsent(name,
                            ignored -> new YasmTuner("antifraud", name.replace('/', '-'))),
                    new SolomonTuner(metricRegistryRoot, config),
                    uaTraitsTuner,
                    currenciesRatesTuner,
                    new DefaultsTuner(prototypesConfig.luaDefaults()),
                    new JsonTuner(prototypesConfig.root()));

            channelsByKey.put(name, channel);
        }

        for (Channel channel : channelsByKey.values()) {
            final ImmutableChannelConfig config = channel.getConfig();
            final List<ImmutableChannelConfig> crossChannels;
            if (!config.crossChannels().isEmpty()) {
                crossChannels = new ArrayList<>(config.crossChannels().size());
                for (String crossChannel : config.crossChannels()) {
                    crossChannels.add(channelsByKey.get(crossChannel).getConfig());
                }
            } else {
                crossChannels = Collections.emptyList();
            }

            final ChannelWithCrossChannels channelWithCrossChannels = new ChannelWithCrossChannels(channel,
                    crossChannels);

            channels.put(config, channelWithCrossChannels);
        }
    }

    @Nonnull
    public ChannelWithCrossChannels getChannel(@Nonnull ScoringData baseData) throws UnknownChannelException {
        final ImmutableChannelConfig channelConfig = resolve(baseData);
        final ChannelWithCrossChannels channel = channels.getOrDefault(channelConfig, null);
        if (channel == null) {
            throw new UnknownChannelException("cannot resolve " + channelConfig);
        }
        return channel;
    }

    @Override
    @Nonnull
    public ImmutableChannelConfig resolve(@Nonnull String channel, @Nullable String subChannel) throws UnknownChannelException {
        return prototypesConfig.resolve(channel, subChannel);
    }

    @Override
    @Nonnull
    public ImmutableChannelConfig resolve(@Nonnull String serviceId) throws UnknownChannelException {
        return prototypesConfig.resolve(serviceId);
    }

    public Stream<Channel> getChannels() {
        return channels.values().stream().map(ChannelWithCrossChannels::getChannel);
    }

    @Nonnull
    public FileListsProvider getListsProvider() {
        return listsProvider;
    }

    @Nonnull
    public MetricRegistry metricRegistryRoot() {
        return metricRegistryRoot;
    }
}
