package ru.yandex.antifraud.lua_context_manager;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.lib.TwoArgFunction;
import jse.org.luaj.vm2.lib.jse.CoerceJavaToLua;

import ru.yandex.antifraud.channel.route_config.ImmutableChannelRouteConfig;
import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.function.BiFunction;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

public class SolomonTuner extends LuaPackage {

    private final MetricRegistry metricRegistry;

    private final Map<ResolutionCode, Rate> resolutionStats = new EnumMap<>(ResolutionCode.class);

    public SolomonTuner(@Nonnull MetricRegistry metricRegistryRoot,
                        @Nonnull ImmutableChannelRouteConfig config) {
        super("solomon");

        metricRegistry = metricRegistryRoot.subRegistry(makeLabels(config));

        for (ResolutionCode code : ResolutionCode.values()) {
            final Rate rateMetric = metricRegistry.rate(code.toString());
            resolutionStats.put(code, rateMetric);
        }
    }

    @Nonnull
    public MetricRegistry subRegistry(@Nonnull String periodName,
                                      @Nonnull String fieldName) {
        return metricRegistry.subRegistry(Labels.of(
                "period", periodName,
                "field", fieldName
        ));
    }

    private static Labels makeLabels(@Nonnull ImmutableChannelRouteConfig config) {
        final Labels labels = Labels.of(
                "channel", config.channel(),
                "channel_uri", config.channelUri());

        final String subChannel = config.subChannel();
        if (subChannel != null) {
            return labels.add("sub_channel", subChannel);
        } else {
            return labels;
        }
    }

    @Override
    public void setup(LuaTable pack) {
        pack.set("register_signal", new SignalFunction((sensor, labels) -> new Pusher(metricRegistry.rate(sensor,
                labels))));
        pack.set("rate", new SignalFunction((sensor, labels) -> new Pusher(metricRegistry.rate(sensor, labels))));
        pack.set("igauge", new SignalFunction((sensor, labels) -> new Pusher(metricRegistry.gaugeInt64(sensor,
                labels))));
        pack.set("dgauge", new SignalFunction((sensor, labels) -> new Pusher(metricRegistry.gaugeDouble(sensor,
                labels))));
        pack.set("counter", new SignalFunction((sensor, labels) -> new Pusher(metricRegistry.counter(sensor, labels))));
    }

    public void pushResolution(ResolutionCode code) {
        resolutionStats.get(code).inc();
    }

    static class Pusher {
        final Consumer<Double> consumer;

        Pusher(GaugeDouble consumer) {
            this.consumer = consumer::add;
        }

        Pusher(GaugeInt64 consumer) {
            this.consumer = n -> consumer.add(n.intValue());
        }

        Pusher(Rate consumer) {
            this.consumer = n -> consumer.add(n.intValue());
        }

        Pusher(Counter consumer) {
            this.consumer = n -> consumer.add(n.intValue());
        }

        @LuaBinding
        public void push(double val) {
            consumer.accept(val);
        }
    }

    static class SignalFunction extends TwoArgFunction {
        private final BiFunction<String, Labels, Pusher> pusherSupplier;

        SignalFunction(BiFunction<String, Labels, Pusher> pusherSupplier) {
            this.pusherSupplier = pusherSupplier;
        }

        @Override
        public LuaValue call(LuaValue sensor, LuaValue luaLabels) {
            final Map<String, String> labels = new HashMap<>();

            for (LuaValue labelKey : luaLabels.checktable().keys()) {
                LuaValue labelValue = luaLabels.get(labelKey);

                labels.put(labelKey.checkjstring(), labelValue.checkjstring());
            }

            final Pusher pusher = pusherSupplier.apply(sensor.checkjstring(), Labels.of(labels));

            return CoerceJavaToLua.coerce(pusher);
        }
    }
}
