package ru.yandex.antifraud.lua_context_manager;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.annotation.Nonnull;

import core.org.luaj.vm2.LuaTable;
import core.org.luaj.vm2.LuaValue;
import core.org.luaj.vm2.lib.OneArgFunction;
import jse.org.luaj.vm2.lib.jse.CoerceJavaToLua;

import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.stater.LongAdderStater;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class YasmTuner extends LuaPackage implements Stater {
    private final String prefix;
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock();
    private final Map<String, LongAdderStater> stats = new HashMap<>();

    private final Map<ResolutionCode, LongAdderStater> resolutionsStats = new HashMap<>();

    public YasmTuner(@Nonnull String prefix,
                     @Nonnull String service) {
        super("yasm");
        this.prefix = prefix + '_' + service;

        for (final ResolutionCode resolutionCode : ResolutionCode.values()) {
            resolutionsStats.put(resolutionCode, new LongAdderStater(this.prefix + '_' + resolutionCode.name() +
                    "_dmmm"));
        }
    }

    private static String normalize(String s) {
        return s.replace("[^A-Za-z0-9]+", "_");
    }

    public void addOurResolution(ResolutionCode resolutionCode) {
        resolutionsStats.get(resolutionCode).adder().add(1);
    }

    @Override
    public void setup(LuaTable pack) {
        pack.set("register_signal", new RegisterSignalFunction(prefix));
    }

    @Override
    public <E extends Exception> void stats(StatsConsumer<? extends E> statsConsumer) throws E {
        final Lock lock = rwLock.readLock();

        for (final LongAdderStater stater : resolutionsStats.values()) {
            stater.stats(statsConsumer);
        }

        lock.lock();
        try {
            for (final LongAdderStater stater : stats.values()) {
                stater.stats(statsConsumer);
            }
        } finally {
            lock.unlock();
        }
    }

    static class Pusher {
        final LongAdder adder;

        Pusher(LongAdder adder) {
            this.adder = adder;
        }

        @LuaBinding
        public void push(long val) {
            adder.add(val);
        }
    }

    class RegisterSignalFunction extends OneArgFunction {
        private final String prefix;

        RegisterSignalFunction(@Nonnull String prefix) {
            this.prefix = prefix;
        }

        @Override
        public LuaValue call(LuaValue signalName) {
            final String key = prefix + '_' + normalize(signalName.checkjstring()) + "_dmmm";
            LongAdderStater stater;

            final Lock lock = rwLock.writeLock();
            lock.lock();
            try {
                stater = stats.computeIfAbsent(key, LongAdderStater::new);
            } finally {
                lock.unlock();
            }
            return CoerceJavaToLua.coerce(new Pusher(stater.adder()));
        }
    }
}
