package ru.yandex.antifraud.lua_context_manager.config;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.config.ChannelConfig;
import ru.yandex.antifraud.channel.config.ChannelConfigBuilder;
import ru.yandex.antifraud.channel.config.ChannelConfigDefaults;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.lua.util.JsonUtils;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractPrototypesConfigBuilder
        <T extends AbstractPrototypesConfigBuilder<T>>
        implements PrototypesConfig {
    private Path additionalScriptsDir;
    private List<String> additionalListsDir;
    private Map<String, ChannelConfigBuilder> channels;
    private JsonMap luaDefaults;
    private Path root;

    protected AbstractPrototypesConfigBuilder(final PrototypesConfig config) {
        additionalScriptsDir = config.additionalScriptsDir();
        additionalListsDir = config.additionalListsDir();

        channels = new HashMap<>();
        for (final Map.Entry<String, ? extends ChannelConfig> kv : config.channels().entrySet()) {
            channels.put(kv.getKey(), new ChannelConfigBuilder(kv.getValue()));
        }

        luaDefaults = config.luaDefaults();

        root = config.root();
    }

    protected AbstractPrototypesConfigBuilder(
            IniConfig config,
            Path rulesRoot,
            PrototypesConfig defaults)
            throws ConfigException {
        root = rulesRoot;
        {
            final IniConfig section = config.sectionOrNull("channels");

            {
                final Path path =
                        Optional.ofNullable(
                                section.getString(ChannelConfigBuilder.SCRIPT_DIR, null)
                        ).map(Path::of).orElse(null);
                if (path != null) {
                    if (path.isAbsolute()) {
                        additionalScriptsDir = path;
                    } else {
                        additionalScriptsDir = Path.of(rulesRoot.toString(), path.toString());
                    }
                } else {
                    additionalScriptsDir = defaults.additionalScriptsDir();
                }
            }
            additionalListsDir = ChannelConfigBuilder.LIST_DIR.extract(section, defaults.additionalListsDir())
                    .stream()
                    .map(d -> Path.of(rulesRoot.toString(), d).toString())
                    .collect(Collectors.toList());

            channels = new HashMap<>();
            // channels values
            for (final Map.Entry<String, IniConfig> kv : section.sections().entrySet()) {
                channels.put(kv.getKey(), new ChannelConfigBuilder(rulesRoot, kv.getValue(),
                        ChannelConfigDefaults.INSTANCE));
            }
        }
        {
            final IniConfig luaDefaults = config.section("lua-defaults");
            if (luaDefaults != null) {
                this.luaDefaults = JsonUtils.ini2json(luaDefaults, BasicContainerFactory.INSTANCE);
            } else {
                this.luaDefaults = defaults.luaDefaults();
            }
        }
    }

    @Nullable
    @Override
    public Path additionalScriptsDir() {
        return additionalScriptsDir;
    }

    public void additionalScriptsDir(@Nullable Path additionalScriptsDir) {
        this.additionalScriptsDir = additionalScriptsDir;
    }

    @Nullable
    @Override
    public List<String> additionalListsDir() {
        return additionalListsDir;
    }

    public void additionalListsDir(@Nullable List<String> additionalListsDir) {
        this.additionalListsDir = additionalListsDir;
    }

    @Nonnull
    @Override
    public Map<String, ChannelConfigBuilder> channels() {
        return channels;
    }

    public void channels(@Nonnull Map<String, ChannelConfigBuilder> channels) {
        this.channels = channels;
    }

    @Nonnull
    @Override
    public JsonMap luaDefaults() {
        return luaDefaults;
    }

    public void luaDefaults(final JsonMap luaDefaults) {
        this.luaDefaults = luaDefaults;
    }

    @Nonnull
    @Override
    public Path root() {
        return root;
    }

    public void root(@Nonnull Path root) {
        this.root = root;
    }
}

