package ru.yandex.antifraud.lua_context_manager.config;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ChannelConfig;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.parser.config.ConfigException;

public class ImmutablePrototypesConfig
        implements PrototypesConfig, ChannelResolver {
    private final Path additionalScriptsDir;
    private final List<String> additionalListsDir;
    private final Map<String, ImmutableChannelConfig> channels;
    private final Map<String, ImmutableChannelConfig> serviceIdIndex;
    private final JsonMap luaDefaults;
    private final Path root;

    public ImmutablePrototypesConfig(final PrototypesConfig config)
            throws ConfigException {
        additionalScriptsDir = config.additionalScriptsDir();
        additionalListsDir = config.additionalListsDir();

        channels = new HashMap<>();
        for (final Map.Entry<String, ? extends ChannelConfig> kv : config.channels().entrySet()) {
            channels.put(kv.getKey(), new ImmutableChannelConfig(kv.getValue()));
        }

        luaDefaults = config.luaDefaults();
        root = config.root();

        serviceIdIndex = new HashMap<>();
        for (ImmutableChannelConfig channelConfig : channels.values()) {
            for (String serviceId : channelConfig.serviceIds()) {
                serviceIdIndex.put(serviceId, channelConfig);
            }
        }
    }

    @Nonnull
    @Override
    public Map<String, ImmutableChannelConfig> channels() {
        return channels;
    }


    @Nullable
    @Override
    public Path additionalScriptsDir() {
        return additionalScriptsDir;
    }

    @Nullable
    @Override
    public List<String> additionalListsDir() {
        return additionalListsDir;
    }

    @Override
    @Nonnull
    public JsonMap luaDefaults() {
        return luaDefaults;
    }

    @Nonnull
    @Override
    public Path root() {
        return root;
    }

    @Nonnull
    @Override
    public ImmutableChannelConfig resolve(@Nonnull String channel, @Nullable String subChannel) throws UnknownChannelException {
        for (ImmutableChannelConfig config : channels.values()) {
            if (config.match(channel, subChannel)) {
                return config;
            }
        }
        throw new UnknownChannelException("channel " + channel + '/' + subChannel + " doesn't exist");
    }

    @Nonnull
    @Override
    public ImmutableChannelConfig resolve(@Nonnull String serviceId) throws UnknownChannelException {
        final ImmutableChannelConfig config = serviceIdIndex.getOrDefault(serviceId, null);

        if (config != null) {
            return config;
        } else {
            throw new UnknownChannelException("serviceId " + serviceId + " doesn't exist");
        }
    }

    public static class NotFoundException extends Exception {
        public static final long serialVersionUID = 0;

        public NotFoundException(final Throwable t) {
            super(t);
        }

        public NotFoundException(final String message) {
            super(message);
        }
    }
}

