package ru.yandex.antifraud.rbl;

import java.net.InetAddress;
import java.net.UnknownHostException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;

public class RblData {
    private static final String INFOS = "infos";
    private static final String GEOBASE = "geobase";
    private static final String REGION_INFO = "region_info";
    private static final String ISO_NAME = "iso_name";
    private static final String IP_TRAITS = "ip_traits";
    private static final String IS_YANDEX_NET = "is_yandex_net";

    private final String isoCountry;
    private final boolean isYandexNet;
    @Nonnull
    private final JsonObject source;

    public RblData(@Nonnull JsonObject source) throws JsonException {
        if (source.type() == JsonObject.Type.STRING) {
            isoCountry = source.asString();
            isYandexNet = false;
            this.source = toLegacyJson();
        } else {
            this.source = source;
            final JsonMap geobase = source.asMap().getMap(INFOS).getMap(GEOBASE);

            final String isoName = geobase
                    .getMap(REGION_INFO)
                    .getString(ISO_NAME);
            isoCountry = isoName.length() >= 2 ? isoName.substring(0, 2) : null;

            isYandexNet = geobase.getMap(IP_TRAITS).getBoolean(IS_YANDEX_NET);
        }
    }

    @Nullable
    public static InetAddress filterHost(@Nullable String ip) {
        if (ip == null) {
            return null;
        }

        InetAddress address;
        try {
            address = InetAddress.getByName(ip);
        } catch (UnknownHostException | SecurityException e) {
            return null;
        }

        if (address.isAnyLocalAddress() || address.isLoopbackAddress()) {
            return null;
        } else {
            return address;
        }
    }

    public JsonMap toLegacyJson() {
        final JsonMap geobase = new JsonMap(BasicContainerFactory.INSTANCE);

        geobase.put(REGION_INFO, JsonMap.singletonMap(ISO_NAME, new JsonString(isoCountry)));
        geobase.put(IP_TRAITS, JsonMap.singletonMap(IS_YANDEX_NET, JsonBoolean.valueOf(isYandexNet)));

        return JsonMap.singletonMap(INFOS, JsonMap.singletonMap(GEOBASE, geobase));
    }

    @Nonnull
    public JsonObject toJson() {
        return source;
    }

    @Nullable
    public String getIsoCountry() {
        return isoCountry;
    }

    public boolean isYandexNet() {
        return isYandexNet;
    }
}
