package ru.yandex.antifraud.storage;

import java.time.Duration;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.aggregates.FieldAcceptor;
import ru.yandex.antifraud.aggregates_v2.AggregatedTimeRange;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.storage.query.BinaryOperation;
import ru.yandex.antifraud.storage.query.BinaryQueryNode;
import ru.yandex.antifraud.storage.query.KeyValueQueryNode;
import ru.yandex.antifraud.storage.query.RangeQueryNode;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.QueryConstructor;

public class CollapsedAggregatesSearchRequest implements SearchRequest {
    public static final RequestType REQUEST_TYPE = RequestType.COLLAPSED_AGGRS;

    @Nonnull
    private final ScoringData scoringData;

    @Nonnull
    private final ImmutableChannelConfig channelConfig;

    @Nullable
    private final String text;

    private final int limit;

    public CollapsedAggregatesSearchRequest(@Nonnull ScoringData scoringData,
                                            @Nonnull ImmutableChannelConfig channelConfig) {
        this.scoringData = scoringData;
        this.channelConfig = channelConfig;

        final BinaryQueryNode kvDaysQuery = new BinaryQueryNode(BinaryOperation.OR);

        final Instant b = scoringData.getTimestamp().plus(channelConfig.gapToCollapsedAggrsBegin());
        final Instant e = scoringData.getTimestamp().plus(channelConfig.gapToCollapsedAggrsEnd());


        final int[] limit = new int[]{0};

        for (var entry : channelConfig.aggregatorsConfig().timeRangesByFieldAcceptors().entrySet()) {
            if (entry.getKey().isEmpty(scoringData)) {
                continue;
            }

            final BinaryQueryNode daysQuery = new BinaryQueryNode(BinaryOperation.OR);
            AggregatedTimeRange.rangesToDays(entry.getValue(), scoringData.getTimestamp(), b, e)
                    .visitRanges((rangeStart, rangeEnd) -> {
                        daysQuery.add(new KeyValueQueryNode("txn_day", new RangeQueryNode(rangeStart, rangeEnd)));
                        limit[0] += Duration.between(rangeStart, rangeEnd).toDays();
                    });

            if (!daysQuery.isEmpty()) {
                final FieldAcceptor.FieldAcceptorInstance fieldAcceptorInstance =
                        entry.getKey().materialize(scoringData);
                kvDaysQuery.add(new BinaryQueryNode(
                        BinaryOperation.AND,
                        new KeyValueQueryNode("key_value", fieldAcceptorInstance.keyValue()),
                        daysQuery));
            }
        }

        this.limit = limit[0];

        if (kvDaysQuery.isEmpty()) {
            text = null;
        } else {
            text = new BinaryQueryNode(BinaryOperation.AND,
                    kvDaysQuery,
                    channelConfig.getQuery(),
                    REQUEST_TYPE.query()
            ).encode();
        }
    }


    @Nonnull
    @Override
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public RequestType requestType() {
        return REQUEST_TYPE;
    }

    @Override
    @Nullable
    public Integer limit() {
        return limit;
    }

    @Override
    public boolean setupQuery(@Nonnull QueryConstructor queryConstructor) throws BadRequestException {
        if (text == null) {
            return false;
        }
        queryConstructor.append("text", text);
        queryConstructor.append("extid", scoringData.getExternalId());
        queryConstructor.append("ranges-over-boolean", 1);

        return true;
    }
}
