package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.aggregates.FieldAcceptor;
import ru.yandex.antifraud.aggregates.Stats;
import ru.yandex.antifraud.aggregates.StructuredStat;
import ru.yandex.antifraud.channel.config.ServiceChannelUri;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class CollapsedAggregatesUpdateRequest implements UpdateRequest {
    public static final RequestType REQUEST_TYPE = RequestType.COLLAPSED_AGGRS;

    public static final String AGGRS_FIELD = "aggrs";
    public static final String STRUCTURED_AGGRS_FIELD = "structured_aggrs";

    @Nonnull
    private final String id;
    @Nonnull
    private final ServiceChannelUri serviceChannelUri;
    @Nonnull
    private final FieldAcceptor.FieldAcceptorInstance fieldAcceptor;
    @Nullable
    private final Stats stats;
    @Nullable
    private final StructuredStat structuredStat;
    @Nonnull
    private final Instant date;

    public CollapsedAggregatesUpdateRequest(@Nonnull ServiceChannelUri serviceChannelUri,
                                            @Nonnull FieldAcceptor.FieldAcceptorInstance fieldAcceptor,
                                            @Nullable Stats stats,
                                            @Nullable StructuredStat structuredStat,
                                            @Nonnull Instant date) {
        this.serviceChannelUri = serviceChannelUri;
        this.fieldAcceptor = fieldAcceptor;
        this.stats = stats;
        this.structuredStat = structuredStat;
        this.date = date.truncatedTo(ChronoUnit.DAYS);
        this.id = makeId(serviceChannelUri, fieldAcceptor, date);
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public String annotation() {
        return REQUEST_TYPE.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return serviceChannelUri.getStorageService();
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }

    public static String makeId(@Nonnull ServiceChannelUri serviceChannelUri,
                                @Nonnull FieldAcceptor.FieldAcceptorInstance fieldAcceptor,
                                @Nonnull Instant date) {
        return "txn_collapsed_aggrs_" +
                serviceChannelUri.getChannelUri() + '_' +
                fieldAcceptor.keyValue() + '_' +
                date.truncatedTo(ChronoUnit.DAYS);
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
        throws IOException
    {
        writer.startObject();
        {
            writer.key("id");
            writer.value(id);

            if (stats != null) {
                writer.key("data");
                writer.value(JsonType.DOLLAR.toString(stats.asFullCleanJson()));
            }

            if (structuredStat != null) {
                writer.key("user_context");
                writer.value(JsonType.DOLLAR.toString(structuredStat.serialize(BasicContainerFactory.INSTANCE)));
            }

            writer.key("channel_uri");
            writer.value(serviceChannelUri.getChannelUri());

            writer.key("key_value");
            writer.value(fieldAcceptor.keyValue());

            writer.key("type");
            writer.value(REQUEST_TYPE);

            writer.key("txn_timestamp");
            writer.value(date.toEpochMilli());
        }
        writer.endObject();
    }

    public AsDebugJson asDebugJson() {
        return new AsDebugJson();
    }

    public class AsDebugJson implements JsonValue {
        @Override
        public void writeValue(final JsonWriterBase w) throws IOException {
            w.startObject();

            w.key("id");
            w.value(id);

            w.key("service");
            w.value(serviceChannelUri.getStorageService());

            w.key("prefix");
            w.value(prefix());

            w.key("channel_uri");
            w.value(serviceChannelUri.getChannelUri());

            w.key("key");
            w.value(fieldAcceptor.key());

            w.key("key_value");
            w.value(fieldAcceptor.keyValue());

            if (stats != null) {
                w.key(AGGRS_FIELD);
                w.value(stats.asFullCleanJson());
            }

            if (structuredStat != null) {
                w.key(STRUCTURED_AGGRS_FIELD);
                structuredStat.jsonify(w);
            }

            w.key("txn_timestamp");
            w.value(date.toEpochMilli());

            w.endObject();
        }
    }
}
