package ru.yandex.antifraud.storage;

import java.util.Collection;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.artefacts.PreparedCounters;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.data.Value;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.storage.query.BinaryOperation;
import ru.yandex.antifraud.storage.query.BinaryQueryNode;
import ru.yandex.antifraud.storage.query.KeyValueQueryNode;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.QueryConstructor;

import static ru.yandex.antifraud.storage.CountersUpdateRequest.makeCountersId;


public class CountersSearchRequest implements SearchRequest {
    public static final RequestType REQUEST_TYPE = RequestType.COUNTERS;
    @Nonnull
    private final String service;

    @Nonnull
    private final BinaryQueryNode idsQuery = new BinaryQueryNode(BinaryOperation.OR);

    public CountersSearchRequest(@Nonnull Collection<PreparedCounters.CounterToCheck> counters,
                                 @Nonnull String service) {
        this.service = service;

        if (!counters.isEmpty()) {
            for (PreparedCounters.CounterToCheck counter : counters) {
                idsQuery.add(counter.id());
            }
        }
    }

    public CountersSearchRequest(@Nonnull ScoringData scoringData,
                                 @Nonnull ImmutableChannelConfig channelConfig) {
        this.service = channelConfig.storageService();
        {
            for (Field field : Field.FIELDS_FOR_COUNTERS) {
                final Value value = scoringData.getValue(field);
                for (String fieldValue : value.values()) {
                    final String keyValue = field.fieldName() + '_' + fieldValue;

                    idsQuery.add(makeCountersId(keyValue, channelConfig));
                }
            }
        }
    }

    public void mergeWith(@Nonnull CountersSearchRequest another) {
        idsQuery.add(another.idsQuery);
    }

    @Nonnull
    @Override
    public String service() {
        return service;
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Nonnull
    @Override
    public String get() {
        return Field.GET_ALL;
    }

    @Override
    public boolean setupQuery(@Nonnull QueryConstructor queryConstructor) throws BadRequestException {
        if (idsQuery.isEmpty()) {
            return false;
        }

        final String textQuery = new KeyValueQueryNode("id", idsQuery).encode();

        queryConstructor.append("text", textQuery);

        return true;
    }

    @Override
    @Nonnull
    public RequestType requestType() {
        return REQUEST_TYPE;
    }
}
