package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import core.org.luaj.vm2.LuaValue;

import ru.yandex.antifraud.artefacts.PreparedCounters;
import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.util.QueryHelpers;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.lua.util.JsonUtils;

public class CountersSingleUpdateRequest implements UpdateRequest {
    public static final RequestType REQUEST_TYPE = RequestType.COUNTERS;

    @Nonnull
    private final List<PreparedCounters.CounterToCheck> counters;

    @Nonnull
    final ImmutableChannelConfig config;

    public CountersSingleUpdateRequest(@Nonnull PreparedCounters.CounterToCheck counterToCheck) {
        this.counters = Collections.singletonList(counterToCheck);
        this.config = counterToCheck.channelConfig();
    }

    public CountersSingleUpdateRequest(@Nonnull JsonMap source,
                                       @Nonnull ChannelResolver resolver) throws JsonException,
            UnknownChannelException, BadRequestException {

        config = resolver.resolve(source.getString("channel"),
                source.getString("sub_channel", null));

        @Nullable final JsonList items = source.getListOrNull("items");

        @Nonnull Instant now = Instant.now();

        if (items == null) {
            @Nonnull final String key = source.getString("key");
            @Nonnull final String value = source.getString("value");

            if (key.isBlank() || value.isBlank()) {
                throw new BadRequestException("blank key or value");
            }

            @Nullable final JsonObject meta = source.getMapOrNull("data");

            counters = Collections.singletonList(
                    new PreparedCounters.CounterToCheck(
                            now,
                            key,
                            value,
                            JsonUtils.jsonToLua(meta),
                            config));
        } else {
            counters = new ArrayList<>(items.size());

            for (final JsonObject item : items) {
                @Nonnull final JsonMap subSource = item.asMap();
                @Nonnull final String key = subSource.getString("key");
                @Nonnull final String value = subSource.getString("value");

                if (key.isBlank() || value.isBlank()) {
                    throw new BadRequestException("blank key or value");
                }

                @Nullable final JsonObject meta = subSource.getMapOrNull("data");

                counters.add(
                        new PreparedCounters.CounterToCheck(
                                now,
                                key,
                                value,
                                JsonUtils.jsonToLua(meta),
                                config));
            }
        }
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
            throws IOException {
        for (final PreparedCounters.CounterToCheck counterToCheck : counters) {
            final long epochMilli = counterToCheck.now().toEpochMilli();

            writer.startObject();
            {
                writer.key("id");
                writer.value(counterToCheck.id());

                {
                    final LuaValue meta = counterToCheck.meta();
                    if (meta != null && meta.istable()) {
                        writer.key("data");
                        writer.value(JsonType.DOLLAR.toString(new JsonUtils.LuaAsJson(meta)));
                    }
                }

                writer.key("channel");
                writer.value(counterToCheck.channelConfig().channel());

                writer.key("sub_channel");
                writer.value(counterToCheck.channelConfig().subChannel());

                writer.key("channel_uri");
                writer.value(counterToCheck.channelConfig().channelUri());

                writer.key("key_value");
                writer.value(counterToCheck.keyValue());

                writer.key("key");
                writer.value(counterToCheck.key());

                writer.key("value");
                writer.value(counterToCheck.value());

                QueryHelpers.INSTANCE.setOnCreate(writer, "txn_timestamp", epochMilli);

                QueryHelpers.INSTANCE.setMax(writer, "txn_status_timestamp", epochMilli);

                {
                    final PreparedCounters.SmoothIncrementContext smoothIncrementContext =
                            counterToCheck.smoothIncrementContext();
                    if (smoothIncrementContext != null) {
                        QueryHelpers.INSTANCE.smoothIncrement(
                                writer,
                                "count",
                                "txn_status_timestamp",
                                counterToCheck.now(),
                                Duration.ofMillis(smoothIncrementContext.getTau()),
                                smoothIncrementContext.getCount());
                    }
                }

                writer.key("type");
                writer.value(REQUEST_TYPE);
            }
            writer.endObject();
        }
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public String annotation() {
        return REQUEST_TYPE.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return config.storageService();
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }
}
