package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.time.Instant;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.channel.route_config.ImmutableChannelRouteConfig;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.data.Value;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.util.QueryHelpers;
import ru.yandex.json.writer.JsonWriterBase;

import static ru.yandex.antifraud.data.Field.FIELDS_FOR_COUNTERS;


public class CountersUpdateRequest implements UpdateRequest {
    public static final RequestType REQUEST_TYPE = RequestType.COUNTERS;
    @Nonnull
    private final ImmutableChannelConfig channelConfig;
    @Nonnull
    ScoringData scoringData;

    public CountersUpdateRequest(@Nonnull ScoringData scoringData,
                                 @Nonnull ImmutableChannelConfig channelConfig) {
        this.scoringData = scoringData;
        this.channelConfig = channelConfig;
    }

    public static String makeCountersId(@Nonnull final String keyValue,
                                        @Nonnull final String channelUri) {
        return "txn_" + channelUri + '_' + keyValue + "_counters";
    }

    public static String makeCountersId(@Nonnull final String key,
                                        @Nonnull final String value,
                                        @Nonnull final String channelUri) {
        return makeCountersId(key + '_' + value, channelUri);
    }

    public static String makeCountersId(@Nonnull final String keyValue,
                                        @Nonnull ImmutableChannelRouteConfig config) {
        return makeCountersId(keyValue, config.channelUri());
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
        throws IOException
    {
        final Instant t = scoringData.getTimestamp();


        for (Field field : FIELDS_FOR_COUNTERS) {
            final String fieldName = field.fieldName();
            final Value fieldValue = scoringData.getValue(field);

            if (fieldValue.isEmpty()) {
                continue;
            }

            for (String value : fieldValue.values()) {
                final String keyValue = fieldName + '_' + value;

                writer.startObject();
                {
                    writer.key("id");
                    writer.value(makeCountersId(keyValue, channelConfig));

                    writer.key("channel");
                    writer.value(scoringData.getChannel());

                    writer.key("sub_channel");
                    writer.value(scoringData.getSubChannel());

                    writer.key("channel_uri");
                    writer.value(channelConfig.channelUri());

                    writer.key("key_value");
                    writer.value(keyValue);

                    writer.key("key");
                    writer.value(fieldName);

                    writer.key("value");
                    writer.value(value);

                    QueryHelpers.INSTANCE.setOnCreate(writer, "txn_timestamp", t.toEpochMilli());

                    QueryHelpers.INSTANCE.setMax(writer, "txn_status_timestamp", t.toEpochMilli());

                    writer.key("type");
                    writer.value(REQUEST_TYPE);

                    writer.key(fieldName);
                    writer.value(value);
                }
                writer.endObject();
            }
        }
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public String annotation() {
        return REQUEST_TYPE.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }
}
