package ru.yandex.antifraud.storage;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.aggregates.Aggregates;
import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.lua.util.JsonUtils;

public class CreateAggregatesRequest implements UpdateRequest {
    public static final RequestType REQUEST_TYPE = RequestType.AGGRS;

    @Nonnull
    private final ScoringData scoringData;
    @Nonnull
    private final Aggregates aggregates;
    @Nullable
    private final Artefacts artefacts;
    @Nonnull
    private final ImmutableChannelConfig channelConfig;

    public CreateAggregatesRequest(@Nonnull ScoringData scoringData,
                                   @Nonnull Aggregates aggregates,
                                   @Nullable Artefacts artefacts,
                                   @Nonnull ImmutableChannelConfig channelConfig) {
        this.scoringData = scoringData;
        this.aggregates = aggregates;
        this.artefacts = artefacts;
        this.channelConfig = channelConfig;
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public String annotation() {
        return REQUEST_TYPE.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
        throws IOException
    {
        writer.startObject();
        {
            writer.key("id");
            writer.value("txn_aggrs_" + scoringData.getExternalId() + '_' + channelConfig.channelUri());

            writer.key("txn_extid");
            writer.value(scoringData.getExternalId());

            writer.key("channel");
            writer.value(scoringData.getChannel());

            writer.key("sub_channel");
            writer.value(scoringData.getSubChannel());

            writer.key("channel_uri");
            writer.value(channelConfig.channelUri());

            writer.key("txn_score_context");
            writer.value(JsonType.DOLLAR.toString(aggregates.asShortJson()));

            if (artefacts != null) {
                if (!artefacts.getListsChecks().isEmpty()) {
                    writer.key("data");
                    writer.value(String.join("\n", artefacts.getListsChecks()));
                }

                writer.key("user_context");
                writer.value(JsonType.DOLLAR.toString(new JsonUtils.LuaAsJson(artefacts.scriptLog())));
            }

            writer.key("type");
            writer.value(REQUEST_TYPE);

            writer.key("txn_timestamp");
            writer.value(scoringData.getTimestamp().toEpochMilli());
        }
        writer.endObject();
    }
}
