package ru.yandex.antifraud.storage;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.data.ListItem;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.storage.query.BinaryOperation;
import ru.yandex.antifraud.storage.query.BinaryQueryNode;
import ru.yandex.antifraud.storage.query.KeyValueQueryNode;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.QueryConstructor;


public class ListItemsSearchRequest implements SearchRequest, Consumer<ListItemSearchRequest> {
    public static final RequestType REQUEST_TYPE = RequestType.FAST_LIST;

    @Nonnull
    private final List<ListItemSearchRequest> itemSearchRequests;
    @Nonnull
    private final String service;


    public ListItemsSearchRequest(@Nonnull String service) {
        this.itemSearchRequests = new ArrayList<>();
        this.service = service;
    }

    @Nonnull
    @Override
    public String service() {
        return service;
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nullable
    public Integer limit() {
        return itemSearchRequests.size();
    }

    @Override
    public boolean setupQuery(@Nonnull QueryConstructor queryConstructor) throws BadRequestException {

        final String textQuery;
        {
            final BinaryQueryNode idsQuery = new BinaryQueryNode(BinaryOperation.OR);

            for (ListItemSearchRequest itemSearchRequest : itemSearchRequests) {
                final String value = itemSearchRequest.value();
                idsQuery.add(ListItem.makeId(itemSearchRequest.channelConfig(),
                        itemSearchRequest.listName(),
                        value));
            }

            if (idsQuery.isEmpty()) {
                return false;
            }

            textQuery = new KeyValueQueryNode(ListItem.ID, idsQuery).encode();
        }
        queryConstructor.append("text", textQuery);

        return true;
    }

    @Override
    @Nonnull
    public RequestType requestType() {
        return REQUEST_TYPE;
    }

    @Override
    public void accept(ListItemSearchRequest listItemSearchRequest) {
        itemSearchRequests.add(listItemSearchRequest);
    }
}
