package ru.yandex.antifraud.storage;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ListItem;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.storage.query.BinaryOperation;
import ru.yandex.antifraud.storage.query.BinaryQueryNode;
import ru.yandex.antifraud.storage.query.KeyValueQueryNode;
import ru.yandex.antifraud.storage.query.QueryNode;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;


public class ListSearchRequest implements SearchRequest {
    public static final RequestType REQUEST_TYPE = RequestType.FAST_LIST;
    private static final int LIMIT = (int) 1e7;

    @Nonnull
    private final ImmutableChannelConfig channelConfig;

    @Nonnull
    private final String listName;

    @Nonnull
    private final Collection<String> values;

    private final int limit;

    public ListSearchRequest(@Nonnull JsonMap rawRequest, @Nonnull ChannelResolver channelResolver) throws JsonException, UnknownChannelException, BadRequestException {
        final String channel = rawRequest.getString("channel", null);
        final String subChannel = rawRequest.getString("sub_channel", null);

        this.channelConfig = channelResolver.resolve(channel, subChannel);

        listName = rawRequest.getString("list_name");
        if (listName.isEmpty()) {
            throw new BadRequestException("list_name is empty");
        }

        final JsonObject jsonValues = rawRequest.get("value");

        switch (jsonValues.type()) {
            case NULL:
                values = Collections.emptyList();
                break;
            case LONG:
            case DOUBLE:
            case STRING:
                values = Collections.singleton(jsonValues.asString());
                break;
            case LIST:
                values = new ArrayList<>(jsonValues.asList().size());
                for (JsonObject v : jsonValues.asList()) {
                    values.add(v.asString());
                }
                break;
            case BOOLEAN:
            case MAP:
            default:
                throw new BadRequestException("unsupported value type " + jsonValues.type());
        }

        for (String value : values) {
            if (value.isEmpty()) {
                throw new BadRequestException("value is empty");
            }
        }

        limit = rawRequest.getInt("limit", 1);
        if (limit < 1 || limit > LIMIT) {
            throw new BadRequestException("limit out of bounds [1:" + LIMIT + ']');
        }
    }

    @Override
    @Nullable
    public Integer limit() {
        return limit;
    }

    @Nonnull
    @Override
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    public boolean setupQuery(@Nonnull QueryConstructor queryConstructor) throws BadRequestException {
        final QueryNode query;

        if (!values.isEmpty()) {
            final BinaryQueryNode ids = new BinaryQueryNode(BinaryOperation.OR);
            for (String value : values) {
                ids.add(ListItem.makeId(channelConfig, listName, value));
            }
            query = new KeyValueQueryNode(ListItem.ID, ids);
        } else {
            query = new BinaryQueryNode(
                    BinaryOperation.AND,
                    new KeyValueQueryNode(ListItem.LIST_NAME, listName),
                    new KeyValueQueryNode(ListItem.TYPE, REQUEST_TYPE.toString()));
        }
        queryConstructor.append("text", query.encode());

        return true;
    }

    @Override
    @Nonnull
    public RequestType requestType() {
        return REQUEST_TYPE;
    }
}
