package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.ListItem;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonWriterBase;

public class ListUpdateRequest implements UpdateRequest {
    public static final RequestType REQUEST_TYPE = RequestType.FAST_LIST;

    @Nonnull
    private final ImmutableChannelConfig channelConfig;
    @Nonnull
    private final String listName;
    @Nonnull
    private final String author;
    @Nonnull
    private final String reason;
    @Nonnull
    private final Collection<String> values;
    @Nonnull
    private final TimeRange timeRange;

    public ListUpdateRequest(@Nonnull ImmutableChannelConfig channelConfig,
                             @Nonnull String listName,
                             @Nonnull String author,
                             @Nonnull String reason,
                             @Nonnull Collection<String> values,
                             @Nonnull TimeRange timeRange) {
        this.channelConfig = channelConfig;
        this.listName = listName;
        this.author = author;
        this.reason = reason;
        this.values = values;
        this.timeRange = timeRange;
    }

    public static ListUpdateRequest parse(@Nonnull JsonMap rawRequest,
                                          @Nonnull ChannelResolver resolver) throws IOException, JsonException,
            TimeRange.IllegalTimeRangeException, BadRequestException, UnknownChannelException {
        final String listName = rawRequest.getString("list_name");
        final String author = rawRequest.getString("author");
        final String channel = rawRequest.getString("channel");
        final String subChannel = rawRequest.getString("sub_channel", null);
        final String reason = rawRequest.getString("reason");

        if (listName.isBlank()) {
            throw new BadRequestException("list_name is empty");
        }

        if (reason.isBlank()) {
            throw new BadRequestException("reason is empty");
        }

        final TimeRange timeRange;
        {
            final Instant from = Instant.ofEpochMilli(rawRequest.getLong("from"));
            final Instant to = Instant.ofEpochMilli(rawRequest.getLong("to"));
            timeRange = new TimeRange(from, to);
        }

        final JsonList rawItems = rawRequest.getList("items");
        final List<String> items = new ArrayList<>(rawItems.size());
        for (JsonObject item : rawItems) {
            items.add(item.asString());
        }

        return new ListUpdateRequest(
                resolver.resolve(channel, subChannel),
                listName,
                author,
                reason,
                items,
                timeRange);
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public String annotation() {
        return REQUEST_TYPE.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
        throws IOException
    {
        for (final String value : values) {
            final String id = ListItem.makeId(channelConfig, listName, value);

            writer.startObject();

            writer.key(ListItem.ID);
            writer.value(id);

            writer.key(ListItem.CHANNEL);
            writer.value(channelConfig.channel());

            writer.key(ListItem.SUB_CHANNEL);
            writer.value(channelConfig.subChannel());

            writer.key(ListItem.CHANNEL_URI);
            writer.value(channelConfig.channelUri());

            writer.key(ListItem.FROM);
            writer.value(timeRange.getStart().toEpochMilli());

            writer.key(ListItem.TO);
            writer.value(timeRange.getEnd().toEpochMilli());

            writer.key(ListItem.LIST_NAME);
            writer.value(listName);

            writer.key(ListItem.AUTHOR);
            writer.value(author);

            writer.key(ListItem.VALUE);
            writer.value(value);

            writer.key(ListItem.TYPE);
            writer.value(RequestType.FAST_LIST);

            writer.key(ListItem.REASON);
            writer.value(reason);

            writer.endObject();
        }
    }

    @Nonnull
    public ImmutableChannelConfig channelConfig() {
        return channelConfig;
    }
}
