package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.concurrent.FailedFuture;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.writer.JsonType;
import ru.yandex.lua.util.JsonUtils;
import ru.yandex.parser.uri.QueryConstructor;

public class SaveClient extends AbstractAsyncClient<SaveClient> {
    @Nonnull
    private final ImmutableHttpHostConfig config;

    public SaveClient(
            @Nonnull final SharedConnectingIOReactor reactor,
            @Nonnull final ImmutableHttpHostConfig config) {
        super(reactor, config);
        this.config = config;
    }

    private SaveClient(
            @Nonnull final CloseableHttpAsyncClient client,
            @Nonnull final SaveClient sample) {
        super(client, sample);
        this.config = sample.config;
    }

    public void delete(
            @Nonnull SearchRequest request,
            @Nonnull Supplier<? extends HttpClientContext> contextGenerator,
            @Nonnull FutureCallback<String> callback) {
        final QueryConstructor queryConstructor;
        try {
            queryConstructor = request.makeDeleteQuery();
            if (queryConstructor == null) {
                callback.completed(null);
                return;
            }
        } catch (BadRequestException e) {
            callback.failed(e);
            return;
        }


        final BasicAsyncRequestProducerGenerator requestProducerGenerator =
                new BasicAsyncRequestProducerGenerator(queryConstructor.toString());

        execute(
                config.host(),
                requestProducerGenerator,
                AsyncStringConsumerFactory.OK,
                contextGenerator,
                callback);
    }

    public Future<String> save(
            @Nonnull UpdateRequest request,
            @Nonnull Supplier<? extends HttpClientContext> contextGenerator,
            FutureCallback<String> callback) {
        final QueryConstructor query;
        try {
            query = request.makeQuery();
        } catch (BadRequestException e) {
            callback.failed(e);
            return new FailedFuture<>(e);
        }

        final BasicAsyncRequestProducerGenerator requestProducerGenerator;

        try {
            requestProducerGenerator =
                    new BasicAsyncRequestProducerGenerator(
                            query.toString(),
                            JsonUtils.compressData(request, JsonType.DOLLAR),
                            ContentType.APPLICATION_JSON);
            requestProducerGenerator.addHeader("Content-Encoding", "gzip");
        } catch (RuntimeException | IOException e) {
            callback.failed(e);
            return new FailedFuture<>(e);
        }

        return execute(
                config.host(),
                requestProducerGenerator,
                AsyncStringConsumerFactory.OK,
                contextGenerator,
                callback);
    }

    @Override
    protected SaveClient adjust(CloseableHttpAsyncClient client) {
        return new SaveClient(client, this);
    }
}
