package ru.yandex.antifraud.storage;

import java.util.List;
import java.util.concurrent.Executor;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.antifraud.invariants.TransactionType;
import ru.yandex.antifraud.util.ExecutedCallbackFutureBase;
import ru.yandex.antifraud.util.JsonParsingCallback;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.UngzippingByteArrayProcessableAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.parser.uri.QueryConstructor;

public class SearchClient extends AbstractAsyncClient<SearchClient> {
    @Nonnull
    private final ImmutableHttpHostConfig config;
    private final int responseLimit;

    @Nonnull
    private final String fieldsToFetch;

    @Nonnull
    private final List<TransactionType> transactionTypesToFetch;

    @Nonnull
    private final Executor executor;

    public SearchClient(
            @Nonnull SharedConnectingIOReactor reactor,
            @Nonnull ImmutableHttpHostConfig config,
            @Nonnull Executor executor,
            @Nonnull String fieldsToFetch,
            int responseLimit,
            @Nonnull List<TransactionType> transactionTypesToFetch) {
        super(reactor, config);
        this.config = config;
        this.fieldsToFetch = fieldsToFetch;
        this.responseLimit = responseLimit;
        this.executor = executor;
        this.transactionTypesToFetch = transactionTypesToFetch;
    }

    private SearchClient(
            @Nonnull final CloseableHttpAsyncClient client,
            @Nonnull final SearchClient sample,
            @Nonnull Executor executor,
            @Nonnull String fieldsToFetch,
            int responseLimit,
            @Nonnull List<TransactionType> transactionTypesToFetch) {
        super(client, sample);
        this.config = sample.config;
        this.fieldsToFetch = fieldsToFetch;
        this.responseLimit = responseLimit;
        this.executor = executor;
        this.transactionTypesToFetch = transactionTypesToFetch;
    }

    @Override
    protected SearchClient adjust(CloseableHttpAsyncClient client) {
        return new SearchClient(client, this, executor, fieldsToFetch, responseLimit,
                transactionTypesToFetch);
    }

    public void search(@Nonnull SearchRequest request,
                       @Nonnull Supplier<? extends HttpClientContext> contextGenerator,
                       @Nonnull FutureCallback<JsonObject> callback) {
        final QueryConstructor queryConstructor;
        try {
            queryConstructor = request.makeSearchQuery();
            if (queryConstructor == null) {
                callback.completed(null);
                return;
            }

            final Integer limit = request.limit() != null ? request.limit() : responseLimit > 0 ? responseLimit : null;
            if (limit != null) {
                queryConstructor.append("length", limit);
            }

        } catch (BadRequestException e) {
            callback.failed(e);
            return;
        }

        final BasicAsyncRequestProducerGenerator requestProducerGenerator =
                new BasicAsyncRequestProducerGenerator(queryConstructor.toString());

        requestProducerGenerator.addHeader("Accept-Encoding", "gzip");

        execute(
                config.host(),
                requestProducerGenerator,
                UngzippingByteArrayProcessableAsyncConsumerFactory.OK,
                contextGenerator,
                new ExecutedCallbackFutureBase<>(executor, new JsonParsingCallback(callback)));
    }

    @Nonnull
    public final String fieldsToFetch() {
        return fieldsToFetch;
    }

    @Nonnull
    public final List<TransactionType> transactionTypesToFetch() {
        return transactionTypesToFetch;
    }
}
