package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.time.Instant;
import java.util.Collection;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.data.Value;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.util.QueryHelpers;
import ru.yandex.json.writer.JsonWriterBase;

import static ru.yandex.antifraud.data.Counters.FIRST_ACQUIRE_SRC;
import static ru.yandex.antifraud.data.Counters.LAST_ACQUIRE_SRC;

public class ServiceVerificationLevelUpdateRequest implements UpdateRequest {
    public static final RequestType REQUEST_TYPE = RequestType.VERIFICATION_LEVEL;

    @Nonnull
    private final String id;
    @Nonnull
    private final Collection<Value> values;
    @Nonnull
    private final Instant t;
    @Nonnull
    private final String storageService;

    private ServiceVerificationLevelUpdateRequest(@Nonnull Collection<Value> values,
                                                  @Nonnull String storageService,
                                                  @Nonnull Instant t) {
        this.id = makeId(values);
        this.values = values;
        this.storageService = storageService;
        this.t = t;
    }

    @Nullable
    public static ServiceVerificationLevelUpdateRequest make(@Nonnull ScoringData scoringData,
                                                             @Nonnull String storageService) {
        final Value verificationLevel = scoringData.getValue(Field.AFS_VERIFICATION_LEVEL);
        final Value status = scoringData.getValue(Field.STATUS);
        if (verificationLevel.isEmpty() || status.isEmpty()) {
            return null;
        }

        return new ServiceVerificationLevelUpdateRequest(
                Stream.of(scoringData.getValue(Field.LOGIN_ID),
                                verificationLevel,
                                scoringData.getValue(Field.SERVICE),
                                status,
                                scoringData.getValue(Field.UID),
                                scoringData.getValue(Field.CARD_ID),
                                scoringData.getValue(Field.DEVICE_ID),
                                scoringData.getValue(Field.USER_AGENT),
                                scoringData.getValue(Field.IS_SECURED))
                        .filter(v -> !v.isEmpty())
                        .collect(Collectors.toList()),
                storageService,
                scoringData.getTimestamp()
        );
    }

    @Nonnull
    public static String makeId(@Nonnull Collection<Value> keyValues) {
        return "txn_verification_level_" + keyValues
                .stream()
                .sorted()
                .map(Value::value)
                .collect(Collectors.joining("_"));
    }

    @Nonnull
    public String id() {
        return id;
    }

    @Override
    public int prefix() {
        return REQUEST_TYPE.storagePrefix();
    }

    @Override
    @Nonnull
    public String annotation() {
        return REQUEST_TYPE.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return storageService;
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
            throws IOException {
        writer.startObject();
        {
            writer.key("id");
            writer.value(id);

            for (Value value : values) {
                writer.key(value.field().fieldName());
                writer.value(value.nonNullValue());
            }

            QueryHelpers.INSTANCE.setOnCreate(writer, FIRST_ACQUIRE_SRC, t.toEpochMilli());
            QueryHelpers.INSTANCE.setMax(writer, LAST_ACQUIRE_SRC, t.toEpochMilli());

            writer.key("type");
            writer.value(REQUEST_TYPE);
        }
        writer.endObject();
    }
}
