package ru.yandex.antifraud.storage;


import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Nonnull;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.antifraud.aggregates.AggregatesBatch;
import ru.yandex.antifraud.artefacts.PreparedCounters;
import ru.yandex.antifraud.artefacts.PreparedLists;
import ru.yandex.antifraud.channel.ChannelWithCrossChannels;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.CollapsedAggregatesResponse;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ListItem;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.lua_context_manager.TimeRange;
import ru.yandex.antifraud.storage.config.ImmutableStorageConfig;
import ru.yandex.antifraud.util.AsyncClientHostRegistrar;
import ru.yandex.antifraud.util.ExecutedCallbackFutureBase;
import ru.yandex.antifraud.util.JoinStringsCallback;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.PayloadFutureCallback;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;


public class StorageClient {
    @Nonnull
    private final Map<String, SearchClient> searchClients = new HashMap<>();

    @Nonnull
    private final Map<String, SaveClient> saveClients = new HashMap<>();

    @Nonnull
    private final Map<String, List<Field>> fieldsToFetch = new HashMap<>();

    @Nonnull
    private final Executor executor;

    public StorageClient(
            @Nonnull final String prefix,
            @Nonnull final SharedConnectingIOReactor reactor,
            @Nonnull final Executor executor,
            @Nonnull final Map<String, ImmutableStorageConfig> configsByServices,
            @Nonnull final AsyncClientHostRegistrar asyncClientRegistrar) {

        this.executor = executor;

        for (Map.Entry<String, ImmutableStorageConfig> kv : configsByServices.entrySet()) {
            final String service = kv.getKey();
            final ImmutableStorageConfig config = kv.getValue();
            final String fieldsToFetch = Field.makeGetParam(config.fieldsToFetch());
            searchClients.put(
                    service,
                    asyncClientRegistrar.registerClientHost(
                            prefix + "StorageAggregationClient" + service,
                            new SearchClient(
                                    reactor,
                                    config.searchConfig(),
                                    executor,
                                    fieldsToFetch,
                                    config.responseLimit(),
                                    config.transactionTypesToFetch()),
                            config.searchConfig()));
            {
                final ImmutableHttpHostConfig saveConfig = config.saveConfig();
                if (saveConfig != null) {
                    saveClients.put(
                            service,
                            asyncClientRegistrar.registerClientHost(
                                    prefix + "StorageSaveClient" + service,
                                    new SaveClient(reactor, saveConfig),
                                    saveConfig));
                }
            }

            this.fieldsToFetch.put(service, config.fieldsToFetch());
        }
    }

    public void save(
            @Nonnull String extid,
            @Nonnull Collection<UpdateRequest> requests,
            @Nonnull RequestsListener listener,
            @Nonnull HttpContext httpContext,
            @Nonnull FutureCallback<String> callback
    ) {
        final Map<UpdateRequest.Key, List<UpdateRequest>> requestsByKeys = new HashMap<>();

        for (UpdateRequest request : requests) {
            requestsByKeys
                    .computeIfAbsent(request.key(), ignored -> new ArrayList<>())
                    .add(request);
        }

        final JoinStringsCallback callbackProxy = new JoinStringsCallback(callback);
        final MultiFutureCallback<String> multiFutureCallback = new MultiFutureCallback<>(callbackProxy);

        for (var entry : requestsByKeys.entrySet()) {
            final List<UpdateRequest> consistentRequests = entry.getValue();

            save(new MultiRequest(extid, entry.getKey(), consistentRequests),
                    listener,
                    httpContext,
                    new ErrorSuppressingFutureCallback<>(
                            multiFutureCallback.newCallback(),
                            (String) null));
        }
        multiFutureCallback.done();
    }

    public void save(
            @Nonnull UpdateRequest request,
            @Nonnull RequestsListener listener,
            @Nonnull HttpContext httpContext,
            @Nonnull FutureCallback<String> callback
    ) {
        try {
            SaveClient client = saveClients.getOrDefault(request.service(), null);
            if (client == null) {
                throw new UnknownServiceException(request.service(), searchClients.keySet());
            }

            client = client.adjust(httpContext);

            client.save(
                    request,
                    listener.createContextGeneratorFor(client),
                    callback);
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    public void delete(
            @Nonnull SearchRequest request,
            @Nonnull RequestsListener listener,
            @Nonnull HttpCoreContext httpCoreContext,
            @Nonnull FutureCallback<String> callback) {
        try {
            SaveClient client = saveClients.getOrDefault(request.service(),
                    null);
            if (client == null) {
                throw new UnknownServiceException(request.service(), searchClients.keySet());
            }

            client = client.adjust(httpCoreContext);

            client.delete(
                    request,
                    listener.createContextGeneratorFor(client),
                    callback);
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    public void search(
            @Nonnull ChannelWithCrossChannels channelWithCrossChannels,
            @Nonnull ScoringData scoringData,
            @Nonnull PreparedLists preparedLists,
            @Nonnull PreparedCounters preparedCounters,
            @Nonnull RequestsListener listener,
            @Nonnull HttpContext httpContext,
            final boolean isSave,
            @Nonnull FutureCallback<List<
                    Map.Entry<ImmutableChannelConfig, AggregatesBatch>>> callback,
            @Nonnull Logger logger) {


        final MultiFutureCallback<
                Map.Entry<ImmutableChannelConfig,
                        AggregatesBatch>> multiFutureCallback = new MultiFutureCallback<>(callback);

        try {
            search(
                    channelWithCrossChannels.getChannel().getConfig(),
                    scoringData,
                    preparedLists.getSearchRequests().values(),
                    preparedCounters.getCountersToCheck(),
                    listener,
                    httpContext,
                    isSave,
                    new PayloadFutureCallback<>(
                            channelWithCrossChannels.getChannel().getConfig(),
                            multiFutureCallback.newCallback()
                    ),
                    logger);

            for (ImmutableChannelConfig crossChannel : channelWithCrossChannels.getCrossChannelsConfigs()) {
                search(
                        crossChannel,
                        scoringData,
                        Collections.emptyList(),
                        Collections.emptyMap(),
                        listener,
                        httpContext,
                        isSave,
                        new PayloadFutureCallback<>(
                                crossChannel,
                                multiFutureCallback.newCallback()
                        ),
                        logger);
            }
        } finally {
            multiFutureCallback.done();
        }
    }

    public void search(
            @Nonnull ImmutableChannelConfig channelConfig,
            @Nonnull ScoringData scoringData,
            @Nonnull Collection<ListItemsSearchRequest> preparedLists,
            @Nonnull Map<String, List<PreparedCounters.CounterToCheck>> countersToCheck,
            @Nonnull RequestsListener listener,
            @Nonnull HttpContext httpContext,
            final boolean isSave,
            @Nonnull FutureCallback<AggregatesBatch> callback,
            @Nonnull Logger logger) {
        final MultiFutureCallback<Consumer<AggregatesBatch>> callbacks =
                new MultiFutureCallback<>(new AggregatesBatchSetupCallback(callback));

        try {
            {
                if (!isSave) {
                    final Supplier<FutureCallback<JsonObject>> transactionsCallback = () ->
                            new ErrorSuppressingFutureCallback<>(
                                    new ExecutedCallbackFutureBase<>(
                                            executor,
                                            new AggregatesBatchSingleSetupCallback(callbacks.newCallback(), entry -> {
                                                final JsonObject src = entry.getValue();
                                                if (src != null) {
                                                    try {
                                                        entry.getKey().updateAggregatedData(src);
                                                    } catch (JsonException e) {
                                                        logger.log(Level.WARNING, "fail to parse aggrs", e);
                                                    }
                                                }
                                            })), (JsonObject) null);

                    final SearchClient client = searchClients.get(channelConfig.storageService());

                    for (Field field : TransactionCreateRequest.FIELDS_FOR_DAILY_IDS) {
                        search(
                                TransactionSearchRequest.today(
                                        scoringData,
                                        client.transactionTypesToFetch(),
                                        channelConfig,
                                        client.fieldsToFetch(),
                                        field),
                                listener,
                                httpContext,
                                transactionsCallback.get());
                        search(
                                TransactionSearchRequest.yesterday(
                                        scoringData,
                                        client.transactionTypesToFetch(),
                                        channelConfig,
                                        client.fieldsToFetch(),
                                        field),
                                listener,
                                httpContext,
                                transactionsCallback.get());
                    }
                }

                if (!isSave) {
                    search(
                            new CollapsedAggregatesSearchRequest(scoringData, channelConfig),
                            listener,
                            httpContext,
                            new ErrorSuppressingFutureCallback<>(
                                    new ExecutedCallbackFutureBase<>(executor,
                                            new AggregatesBatchSingleSetupCallback(callbacks.newCallback(), entry -> {
                                                final JsonObject src = entry.getValue();
                                                if (src != null) {
                                                    try {
                                                        entry.getKey().setCollapsedAggregatesResponse(new CollapsedAggregatesResponse(src.asMap()));
                                                    } catch (JsonException e) {
                                                        logger.log(Level.WARNING, "fail to parse collapsed aggrs", e);
                                                    }
                                                }
                                            })), (JsonObject) null));
                }

                final Supplier<FutureCallback<JsonObject>> updateCountersCallbackSupplier =
                        () -> new ErrorSuppressingFutureCallback<>(
                                new ExecutedCallbackFutureBase<>(executor,
                                        new AggregatesBatchSingleSetupCallback(callbacks.newCallback(),
                                                responseEntry -> {
                                                    final JsonObject src = responseEntry.getValue();
                                                    if (src != null) {
                                                        try {
                                                            responseEntry.getKey().updateCounters(src.asMap());
                                                        } catch (JsonException e) {
                                                            logger.log(Level.WARNING, "fail to parse counters", e);
                                                        }
                                                    }
                                                })), (JsonObject) null);

                {
                    final Map<String, CountersSearchRequest> requestsByService = new HashMap<>();
                    {
                        final CountersSearchRequest request = new CountersSearchRequest(scoringData, channelConfig);
                        requestsByService.put(request.service(), request);
                    }

                    for (Map.Entry<String, List<PreparedCounters.CounterToCheck>> entry : countersToCheck.entrySet()) {
                        final CountersSearchRequest request = new CountersSearchRequest(entry.getValue(),
                                entry.getKey());
                        requestsByService.merge(request.service(), request, (oldRequest, newRequest) -> {
                            oldRequest.mergeWith(newRequest);
                            return oldRequest;
                        });
                    }

                    for (CountersSearchRequest request : requestsByService.values()) {
                        search(
                                request,
                                listener,
                                httpContext,
                                updateCountersCallbackSupplier.get());
                    }
                }

                final Supplier<FutureCallback<JsonObject>> addListsCallbackSupplier =
                        () -> new ErrorSuppressingFutureCallback<>(
                                new ExecutedCallbackFutureBase<>(executor,
                                        new AggregatesBatchSingleSetupCallback(callbacks.newCallback(), entry -> {
                                            final JsonObject src = entry.getValue();
                                            if (src != null) {
                                                try {
                                                    entry.getKey().addLists(ListItem.parseResponse(src));
                                                } catch (JsonException | TimeRange.IllegalTimeRangeException e) {
                                                    logger.log(Level.WARNING, "fail to parse lists", e);
                                                }
                                            }
                                        })), (JsonObject) null);
                for (ListItemsSearchRequest request : preparedLists) {
                    search(request,
                            listener,
                            httpContext,
                            addListsCallbackSupplier.get());
                }
            }
        } finally {
            callbacks.done();
        }
    }

    public void search(@Nonnull SearchRequest request,
                       @Nonnull RequestsListener listener,
                       @Nonnull HttpContext context,
                       @Nonnull FutureCallback<JsonObject> callback) {
        try {
            SearchClient client = searchClients.getOrDefault(request.service(), null);
            if (client == null) {
                throw new UnknownServiceException(request.service(), searchClients.keySet());
            }

            client = client.adjust(context);

            client.search(
                    request,
                    listener.createContextGeneratorFor(client),
                    callback);
        } catch (Exception e) {
            callback.failed(e);
        }
    }

    @Nonnull
    public Map<String, List<Field>> fieldsToFetch() {
        return fieldsToFetch;
    }

    private static class AggregatesBatchSetupCallback extends CallbackFutureBase<AggregatesBatch,
            List<Consumer<AggregatesBatch>>> {
        @Nonnull
        private final AggregatesBatch aggregatesBatch = new AggregatesBatch();

        protected AggregatesBatchSetupCallback(@Nonnull FutureCallback<? super AggregatesBatch> callback) {
            super(callback);
        }

        @Override
        protected AggregatesBatch convertResult(List<Consumer<AggregatesBatch>> consumers) {
            for (Consumer<AggregatesBatch> consumer : consumers) {
                consumer.accept(aggregatesBatch);
            }
            return aggregatesBatch;
        }
    }

    private static class AggregatesBatchSingleSetupCallback extends CallbackFutureBase<Consumer<AggregatesBatch>,
            JsonObject> {

        @Nonnull
        private final Consumer<Map.Entry<AggregatesBatch, JsonObject>> settuper;

        protected AggregatesBatchSingleSetupCallback(FutureCallback<? super Consumer<AggregatesBatch>> callback,
                                                     @Nonnull Consumer<Map.Entry<AggregatesBatch, JsonObject>> settuper) {
            super(callback);
            this.settuper = settuper;
        }

        @Override
        protected Consumer<AggregatesBatch> convertResult(JsonObject result) {
            return aggregatesBatch -> {
                if (result != null) {
                    settuper.accept(new AbstractMap.SimpleEntry<>(aggregatesBatch, result));
                }
            };
        }
    }
}

