package ru.yandex.antifraud.storage;

import java.io.IOException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.artefacts.Artefacts;
import ru.yandex.antifraud.artefacts.Resolution;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.channel.route_config.ImmutableChannelRouteConfig;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.invariants.TransactionType;
import ru.yandex.antifraud.util.Shard;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriterBase;

public class TransactionCreateRequest implements UpdateRequest {
    public static final String DAILY_ID_SUFFIX = "_daily_id";
    public static final List<Field> FIELDS_FOR_DAILY_IDS = Arrays.asList(
            Field.UID,
            Field.CARD_ID,
            Field.IP,
            Field.USER_PHONE
    );

    @Nonnull
    private final ScoringData scoringData;
    @Nonnull
    private final List<Field> fieldsToFetch;
    @Nullable
    private final Artefacts artefacts;
    @Nonnull
    private final ImmutableChannelConfig channelConfig;

    @Nonnull
    private final RequestType requestType;

    public TransactionCreateRequest(@Nonnull ScoringData scoringData,
                                    @Nonnull List<Field> fieldsToFetch,
                                    @Nullable Artefacts artefacts,
                                    @Nonnull ImmutableChannelConfig channelConfig,
                                    @Nonnull RequestType requestType) {
        this.scoringData = scoringData;
        this.fieldsToFetch = fieldsToFetch;
        this.artefacts = artefacts;
        this.channelConfig = channelConfig;
        this.requestType = requestType;
    }

    @Override
    public int prefix() {
        return (int) Shard.calcByTimestamp(scoringData.getTimestamp());
    }

    @Override
    @Nonnull
    public String annotation() {
        return requestType.toString();
    }

    @Override
    @Nonnull
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public boolean addIfNotExists() {
        return true;
    }

    @Nonnull
    public static String makeDailyId(
            @Nonnull String id,
            @Nonnull ImmutableChannelRouteConfig config,
            @Nonnull TransactionType transactionType,
            @Nonnull Instant timestamp) {

        return id + '_' +
                config.channelUri() + '_' +
                transactionType + '_' +
                timestamp.truncatedTo(ChronoUnit.DAYS).toEpochMilli();
    }

    @Nonnull
    public static String makeId(@Nonnull String externalId,
                                @Nonnull String channelUri,
                                @Nonnull RequestType requestType) {
        String id = "txn_";
        if (requestType != RequestType.MAIN) {
            id += requestType.toString() + '_';
        }
        return id + externalId + '_' + channelUri;
    }

    @Override
    public void writeDocs(@Nonnull final JsonWriterBase writer)
        throws IOException
    {
        writer.startObject();
        {
            writer.key("id");
            writer.value(makeId(scoringData.getExternalId(), channelConfig.channelUri(), requestType));

            writer.key("channel_uri");
            writer.value(channelConfig.channelUri());

            writer.key("data");
            writer.value(JsonType.DOLLAR.toString(scoringData.asTruncatedJson()));

            writer.key("type");
            writer.value(requestType);

            writer.key("transaction_type");
            writer.value(scoringData.getTransactionType());

            for (Field field : fieldsToFetch) {
                final String value = scoringData.getValue(field).value();
                if (value != null) {
                    writer.key(field.fieldName());
                    writer.value(value);
                }
            }

            if (scoringData.getTransactionType() != null) {
                for (Field field : FIELDS_FOR_DAILY_IDS) {
                    final String value = scoringData.getValue(field).value();
                    if (value != null) {
                        writer.key(field + DAILY_ID_SUFFIX);
                        writer.value(
                                makeDailyId(
                                        value,
                                        channelConfig,
                                        scoringData.getTransactionType(),
                                        scoringData.getTimestamp()));
                    }
                }
            }

            if (artefacts != null) {
                final Resolution resolution = artefacts.getResolution();

                writer.key("txn_afs_action");
                writer.value(resolution.getResolutionCode());

                writer.key("txn_afs_tags");
                writer.value(JsonType.NORMAL.toString(resolution.getTags()));

                writer.key("txn_afs_reason");
                writer.value(String.join("\n", resolution.getReason()));

                writer.key("txn_afs_queue");
                writer.value(artefacts.getQueues());

                writer.key("txn_afs_comment");
                writer.value(artefacts.getComments());
            }
        }
        writer.endObject();
    }
}
