package ru.yandex.antifraud.storage;

import java.time.Duration;
import java.time.Instant;
import java.util.List;

import javax.annotation.Nonnull;

import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.invariants.TransactionType;
import ru.yandex.antifraud.storage.query.BinaryOperation;
import ru.yandex.antifraud.storage.query.BinaryQueryNode;
import ru.yandex.antifraud.storage.query.KeyValueQueryNode;
import ru.yandex.antifraud.util.Shard;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.QueryConstructor;

import static ru.yandex.antifraud.storage.TransactionCreateRequest.DAILY_ID_SUFFIX;
import static ru.yandex.antifraud.storage.TransactionCreateRequest.makeDailyId;

public class TransactionSearchRequest implements SearchRequest {
    public static final RequestType REQUEST_TYPE = RequestType.MAIN;

    @Nonnull
    private final ScoringData scoringData;

    @Nonnull
    private final ImmutableChannelConfig channelConfig;

    @Nonnull
    private final List<TransactionType> transactionTypesToFetch;

    @Nonnull
    private final String fieldsToFetch;

    private final int prefix;

    @Nonnull
    private final Instant timestamp;

    @Nonnull
    private final Field field;

    private TransactionSearchRequest(@Nonnull ScoringData scoringData,
                                     @Nonnull List<TransactionType> transactionTypesToFetch,
                                     @Nonnull ImmutableChannelConfig channelConfig,
                                     @Nonnull String fieldsToFetch,
                                     int prefix,
                                     @Nonnull Instant timestamp,
                                     @Nonnull Field field) {
        this.scoringData = scoringData;
        this.channelConfig = channelConfig;
        this.transactionTypesToFetch = transactionTypesToFetch;
        this.fieldsToFetch = fieldsToFetch;
        this.prefix = prefix;
        this.timestamp = timestamp;
        this.field = field;
    }

    @Nonnull
    public static TransactionSearchRequest today(@Nonnull ScoringData scoringData,
                                                 @Nonnull List<TransactionType> transactionTypesToFetch,
                                                 @Nonnull ImmutableChannelConfig channelConfig,
                                                 @Nonnull String fieldsToFetch,
                                                 @Nonnull Field field) {
        final int prefix = (int) Shard.calcByTimestamp(scoringData.getTimestamp());
        return new TransactionSearchRequest(
                scoringData,
                transactionTypesToFetch,
                channelConfig,
                fieldsToFetch,
                prefix,
                scoringData.getTimestamp(),
                field
        );
    }

    @Nonnull
    public static TransactionSearchRequest yesterday(@Nonnull ScoringData scoringData,
                                                     @Nonnull List<TransactionType> transactionTypesToFetch,
                                                     @Nonnull ImmutableChannelConfig channelConfig,
                                                     @Nonnull String fieldsToFetch,
                                                     @Nonnull Field field) {
        final Instant timestamp = scoringData.getTimestamp().minus(Duration.ofDays(1));
        final int prefix = (int) Shard.calcByTimestamp(timestamp);
        return new TransactionSearchRequest(
                scoringData,
                transactionTypesToFetch,
                channelConfig,
                fieldsToFetch,
                prefix,
                timestamp,
                field
        );
    }


    @Nonnull
    @Override
    public String service() {
        return channelConfig.storageService();
    }

    @Override
    public int prefix() {
        return prefix;
    }

    @Override
    @Nonnull
    public String get() {
        return fieldsToFetch;
    }

    @Override
    public boolean setupQuery(@Nonnull QueryConstructor queryConstructor) throws BadRequestException {
        final String value = scoringData.getValue(field).value();
        if (value == null) {
            return false;
        }

        final BinaryQueryNode dailyIds = new BinaryQueryNode(BinaryOperation.OR);
        for (TransactionType transactionType : transactionTypesToFetch) {
            dailyIds.add(makeDailyId(
                    value,
                    channelConfig,
                    transactionType,
                    timestamp));
        }

        queryConstructor.append("text", new KeyValueQueryNode(field + DAILY_ID_SUFFIX, dailyIds).encode());
        queryConstructor.append("extid", scoringData.getExternalId());

        return true;
    }

    @Override
    @Nonnull
    public RequestType requestType() {
        return REQUEST_TYPE;
    }
}
