package ru.yandex.antifraud.storage;

import java.nio.charset.CharacterCodingException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.channel.ChannelResolver;
import ru.yandex.antifraud.channel.config.ImmutableChannelConfig;
import ru.yandex.antifraud.invariants.RequestType;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.storage.query.BinaryOperation;
import ru.yandex.antifraud.storage.query.BinaryQueryNode;
import ru.yandex.antifraud.storage.query.RawValueQueryNode;
import ru.yandex.antifraud.util.Shard;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;

public class TransactionSearchUIRequest implements SearchRequest {

    @Nonnull
    private final String query;

    private final int limit;

    @Nonnull
    private final String service;

    private final int prefix;

    public TransactionSearchUIRequest(@Nonnull String query, int limit, @Nonnull String service, int prefix) {
        this.query = query;
        this.limit = limit;
        this.service = service;
        this.prefix = prefix;
    }

    public static List<TransactionSearchUIRequest> parse(@Nonnull JsonMap rawRequest,
                                                         @Nonnull ChannelResolver resolver) throws JsonException,
            UnknownChannelException, CharacterCodingException {

        final String channel = rawRequest.getString("channel");
        final String subChannel = rawRequest.getString("sub_channel", null);
        final String rawQuery =
                Optional.ofNullable(rawRequest.getString("query", null))
                        .map(String::strip)
                        .orElse(null);
        final RequestType requestType = RequestType.valueOf(rawRequest.getString("type"));
        Long prefix = rawRequest.getLong("prefix", null);

        final int limit = clamp(rawRequest.getInt("limit", 1), 1, 10000);

        final ImmutableChannelConfig channelConfig = resolver.resolve(channel, subChannel);

        final String service = channelConfig.storageService();
        final String query;
        {
            final BinaryQueryNode queryNode = new BinaryQueryNode(
                    BinaryOperation.AND,
                    requestType.query(),
                    channelConfig.getQuery()
            );

            if (rawQuery != null && !rawQuery.isBlank()) {
                queryNode.add(new RawValueQueryNode(rawQuery));
            }
            query = queryNode.encode();
        }

        if (prefix != null) {
            return Collections.singletonList(new TransactionSearchUIRequest(query, limit, service,
                    Math.toIntExact(prefix)));
        }
        prefix = Shard.calcByTimestamp(Instant.now());

        final int step, count;
        switch (service) {
            case "so_fraud_data":
            case "so_fraud_passport":
                step = -7;
                count = 5;
                break;
            case "so_fraud_testing":
                step = 0;
                count = 1;
                break;
            default:
                throw new UnknownChannelException("unknown service " + service);
        }

        final List<TransactionSearchUIRequest> requests = new ArrayList<>(count);
        for (int i = 0; i < count; i++) {
            requests.add(new TransactionSearchUIRequest(query, limit, service,
                    Math.toIntExact(Shard.normalize(prefix + step * i))));
        }

        return requests;
    }

    private static int clamp(int val, int min, int max) {
        return Math.max(min, Math.min(max, val));
    }

    @Nonnull
    @Override
    public String service() {
        return service;
    }


    @Nullable
    @Override
    public Integer shard() {
        return prefix;
    }

    @Override
    public int prefix() {
        return prefix;
    }

    @Override
    public boolean setupQuery(@Nonnull QueryConstructor queryConstructor) throws BadRequestException {
        queryConstructor.append("backend", "ui");
        queryConstructor.append("text", query);
        queryConstructor.append("sort", "txn_timestamp");
        queryConstructor.append("collector", "pruning(txn_day)");
        queryConstructor.append("ranges-over-boolean", 1);
        queryConstructor.append("early-interrupt", 1);

        return true;
    }

    @Override
    @Nonnull
    public RequestType requestType() {
        return TransactionSearchRequest.REQUEST_TYPE;
    }

    @Override
    @Nullable
    public Integer limit() {
        return limit;
    }
}
