package ru.yandex.antifraud.storage.config;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.data.Field;
import ru.yandex.antifraud.invariants.TransactionType;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public abstract class AbstractStorageConfigBuilder
        <T extends AbstractStorageConfigBuilder<T>>
        implements StorageConfig {

    private int responseLimit;

    @Nonnull
    private HttpHostConfigBuilder searchConfig;

    @Nullable
    private HttpHostConfigBuilder saveConfig;

    @Nonnull
    private List<Field> fieldsToFetch;

    @Nonnull
    private List<TransactionType> transactionTypesToFetch;

    protected AbstractStorageConfigBuilder(final StorageConfig config) {
        responseLimit = config.responseLimit();

        searchConfig = new HttpHostConfigBuilder(config.searchConfig());

        if (config.saveConfig() != null) {
            saveConfig = new HttpHostConfigBuilder(config.saveConfig());
        } else {
            saveConfig = null;
        }

        fieldsToFetch = config.fieldsToFetch();

        transactionTypesToFetch = config.transactionTypesToFetch();
    }

    protected AbstractStorageConfigBuilder(
            final IniConfig config,
            final StorageConfig defaults)
            throws ConfigException {
        responseLimit = config.getInt("response-limit", defaults.responseLimit());

        {
            final IniConfig section = config.section("search");
            searchConfig = new HttpHostConfigBuilder(section, defaults.searchConfig());
        }
        {
            final IniConfig section = config.sectionOrNull("save");
            if (section != null) {
                saveConfig = new HttpHostConfigBuilder(section, defaults.saveConfig());
            } else {
                saveConfig = null;
            }
        }

        fieldsToFetch = FIELDS_TO_FETCH.extract(config, defaults.fieldsToFetch());
        transactionTypesToFetch = TRANSACTION_TYPES_TO_FETCH.extract(config, defaults.transactionTypesToFetch());
    }

    @Override
    public int responseLimit() {
        return responseLimit;
    }

    public void responseLimit(int limit) {
        responseLimit = limit;
    }

    @Override
    @Nonnull
    public HttpHostConfigBuilder searchConfig() {
        return searchConfig;
    }

    public void searchConfig(@Nonnull HttpHostConfigBuilder searchConfig) {
        this.searchConfig = searchConfig;
    }

    @Override
    @Nullable
    public HttpHostConfigBuilder saveConfig() {
        return saveConfig;
    }

    public void saveConfig(@Nonnull HttpHostConfigBuilder saveConfig) {
        this.saveConfig = saveConfig;
    }

    @Override
    @Nonnull
    public List<Field> fieldsToFetch() {
        return fieldsToFetch;
    }

    public void fieldsToFetch(@Nonnull List<Field> fieldsToFetch) {
        this.fieldsToFetch = fieldsToFetch;
    }

    @Override
    @Nonnull
    public List<TransactionType> transactionTypesToFetch() {
        return transactionTypesToFetch;
    }

    public void transactionTypesToFetch(@Nonnull List<TransactionType> transactionTypesToFetch) {
        this.transactionTypesToFetch = transactionTypesToFetch;
    }
}

