package ru.yandex.antifraud.storage.query;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

public class BinaryQueryNode implements QueryNode {
    @Nonnull
    final BinaryOperation operation;
    @Nonnull
    private final List<QueryNode> nodes = new ArrayList<>();

    public BinaryQueryNode(@Nonnull BinaryOperation operation) {
        this.operation = operation;
    }

    @SafeVarargs
    public <T extends QueryNode> BinaryQueryNode(@Nonnull BinaryOperation operation,
                                                 @Nonnull T... nodes) {
        this.operation = operation;
        addAll(Arrays.asList(nodes));
    }

    @SafeVarargs
    public <T extends String> BinaryQueryNode(@Nonnull BinaryOperation operation,
                                              @Nonnull T... nodes) {
        this.operation = operation;
        addAll(Arrays.stream(nodes).map(StringValueQueryNode::new).collect(Collectors.toList()));
    }

    public BinaryQueryNode(@Nonnull BinaryOperation operation,
                           @Nonnull Collection<? extends String> nodes) {
        this.operation = operation;
        addAllStrings(nodes);
    }

    @SafeVarargs
    public <T extends Long> BinaryQueryNode(@Nonnull BinaryOperation operation,
                                            @Nonnull T... nodes) {
        this.operation = operation;
        addAll(Arrays.stream(nodes).map(LongValueQueryNode::new).collect(Collectors.toList()));
    }

    @Override
    public void encode(StringBuilder textQuery,
                       boolean isRoot) {
        final boolean needBraces = !isRoot && nodes.size() > 1;
        if (needBraces) {
            textQuery.append('(');
        }

        boolean isFirstNode = true;
        for (QueryNode node : nodes) {
            if (!isFirstNode) {
                if (operation != BinaryOperation.OR) {
                    textQuery.append(' ');
                    textQuery.append(operation);
                }
                textQuery.append(' ');
            }
            isFirstNode = false;
            node.encode(textQuery, false);
        }

        if (needBraces) {
            textQuery.append(')');
        }
    }

    public BinaryQueryNode add(@Nonnull QueryNode node) {
        if (node instanceof BinaryQueryNode && ((BinaryQueryNode) node).operation == operation) {
            nodes.addAll(((BinaryQueryNode) node).nodes);
        } else {
            nodes.add(node);
        }
        return this;
    }

    public BinaryQueryNode addAll(@Nonnull Collection<QueryNode> nodes) {
        for (QueryNode node : nodes) {
            add(node);
        }
        return this;
    }

    public BinaryQueryNode addAllStrings(@Nonnull Collection<? extends String> nodes) {
        for (String node : nodes) {
            add(node);
        }
        return this;
    }

    public BinaryQueryNode add(@Nonnull String value) {
        nodes.add(new StringValueQueryNode(value));
        return this;
    }

    public BinaryQueryNode add(long value) {
        nodes.add(new LongValueQueryNode(value));
        return this;
    }

    public boolean isEmpty() {
        return nodes.isEmpty();
    }
}
