package ru.yandex.antifraud_runner;


import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.antifraud.aggregates.AggregatesScoringContext;
import ru.yandex.antifraud.aggregates.Aggregator;
import ru.yandex.antifraud.aggregates.FieldAcceptor;
import ru.yandex.antifraud.aggregates.MultiStructuredStat;
import ru.yandex.antifraud.aggregates.Stats;
import ru.yandex.antifraud.aggregates.StructuredAggregator;
import ru.yandex.antifraud.aggregates.StructuredStat;
import ru.yandex.antifraud.aggregates_v2.AggregatedTimeRange;
import ru.yandex.antifraud.aggregates_v2.AggregatorConsumer;
import ru.yandex.antifraud.aggregates_v2.ConglomerateAggregatorConsumer;
import ru.yandex.antifraud.aggregates_v2.aggregate_config.AggregatorConfigBuilder;
import ru.yandex.antifraud.aggregates_v2.aggregates_config.AggregatorsConfigBuilder;
import ru.yandex.antifraud.aggregates_v2.aggregates_config.ImmutableAggregatorsConfig;
import ru.yandex.antifraud.channel.ExecutionContext;
import ru.yandex.antifraud.channel.config.ServiceChannelUri;
import ru.yandex.antifraud.data.ScoringData;
import ru.yandex.antifraud.invariants.ResolutionCode;
import ru.yandex.antifraud.invariants.TransactionStatus;
import ru.yandex.antifraud.lua_context_manager.UnknownChannelException;
import ru.yandex.antifraud.rbl.RblData;
import ru.yandex.antifraud.storage.CollapsedAggregatesUpdateRequest;
import ru.yandex.antifraud.util.ConsumerSafe;


class LogAggregator implements LogItemConsumer<UnknownChannelException> {
    @Nonnull
    private final ConsumerSafe<CollapsedAggregatesUpdateRequest, Exception> requestsConsumer;
    private final Map<ServiceChannelUri, ConglomerateAggregatorConsumer> consumers = new HashMap<>();
    private Instant earliestTimestamp = Instant.MAX;
    private Instant latestTimestamp = Instant.MIN;

    LogAggregator(@Nonnull ConsumerSafe<CollapsedAggregatesUpdateRequest, Exception> requestsConsumer) {
        this.requestsConsumer = requestsConsumer;
    }

    private static final ImmutableAggregatorsConfig AGGREGATOR_CONFIG;

    static {
        final AggregatorConfigBuilder aggregatorConfigBuilder = new AggregatorConfigBuilder();
        aggregatorConfigBuilder.timeRanges(Collections.singleton(AggregatedTimeRange.ANY));
        aggregatorConfigBuilder.fieldAcceptors(Arrays.stream(FieldAcceptor.values()).collect(Collectors.toSet()));
        aggregatorConfigBuilder.aggregators(Arrays.stream(Aggregator.values()).collect(Collectors.toSet()));
        aggregatorConfigBuilder.structuredAggregators(Arrays.stream(StructuredAggregator.values()).collect(Collectors.toSet()));

        final AggregatorsConfigBuilder aggregatorsConfigBuilder = new AggregatorsConfigBuilder();
        aggregatorsConfigBuilder.aggregates(Collections.singletonList(aggregatorConfigBuilder.build()));

        AGGREGATOR_CONFIG = aggregatorsConfigBuilder.build();
    }

    @Override
    public void accept(ExecutionContext executionContext) {
        @Nonnull final ScoringData request = executionContext.getScoreData();
        @Nullable final Optional<RblData> rblData = Optional.ofNullable(executionContext.getRblData());
        @Nullable final String ipIsoCountry = rblData.map(RblData::getIsoCountry).orElse(null);
        @Nullable final Optional<ScoringData> upData = Optional.ofNullable(executionContext.getSaveData());
        @Nullable final TransactionStatus transactionStatus =
                upData.map(ScoringData::getTransactionStatus).orElse(null);
        @Nullable final ResolutionCode resolutionCode =
                upData.map(ScoringData::getResolutionCode).orElse(null);
        @Nullable final Boolean isAuthed = upData.map(ScoringData::isAuthed).map(v -> v != 0).orElse(false);
        @Nullable final Collection<String> queues = executionContext.getQueues();

        if (request.getTimestamp().isAfter(latestTimestamp)) {
            latestTimestamp = request.getTimestamp();
        }

        if (request.getTimestamp().isBefore(earliestTimestamp)) {
            earliestTimestamp = request.getTimestamp();
        }

        final AggregatesScoringContext aggregatesScoringContext = new AggregatesScoringContext(request,
                ipIsoCountry,
                transactionStatus,
                resolutionCode,
                isAuthed,
                queues);

        final ConglomerateAggregatorConsumer materialized = AGGREGATOR_CONFIG.materialize(aggregatesScoringContext);

        consumers
                .compute(
                        new ServiceChannelUri(
                                executionContext.getChannelUri(),
                                executionContext.getStorageService()),
                        (key, old) -> {
                            if (old == null) {
                                return materialized;
                            } else {
                                old.addAll(materialized.getAggregatorConsumers());
                                return old;
                            }
                        })
                .accept(aggregatesScoringContext);
    }

    @Override
    public void finish() throws Exception {
        final Instant averageDay = getAverageDay();

        for (var entry : consumers.entrySet()) {
            final ServiceChannelUri key = entry.getKey();
            final ConglomerateAggregatorConsumer conglomerateAggregatorConsumer = entry.getValue();

            final Set<FieldAcceptor.FieldAcceptorInstance> fieldAcceptors = new HashSet<>();
            final Map<FieldAcceptor.FieldAcceptorInstance, Stats> statsByField = new HashMap<>();
            final Map<FieldAcceptor.FieldAcceptorInstance, MultiStructuredStat.Instance> structuredStatsByField =
                    new HashMap<>();

            for (AggregatorConsumer aggregatorConsumer : conglomerateAggregatorConsumer.getAggregatorConsumers()) {
                fieldAcceptors.add(aggregatorConsumer.getFieldAcceptorInstance());
                {
                    final Aggregator.AggregatorInstance aggregatorInstance = aggregatorConsumer.getAggregatorInstance();
                    if (aggregatorInstance != null) {
                        statsByField
                                .computeIfAbsent(aggregatorConsumer.getFieldAcceptorInstance(), ignored -> new Stats())
                                .put(aggregatorInstance.ideal(), aggregatorInstance.getStat());
                    }
                }
                {
                    final StructuredAggregator.StructuredAggregatorInstance aggregatorInstance =
                            aggregatorConsumer.getStructuredAggregatorInstance();
                    if (aggregatorInstance != null) {
                        structuredStatsByField
                                .computeIfAbsent(aggregatorConsumer.getFieldAcceptorInstance(),
                                        ignored -> new MultiStructuredStat.Instance())
                                .put(aggregatorInstance.ideal().getName(), aggregatorInstance.getStat());
                    }
                }
            }

            for (FieldAcceptor.FieldAcceptorInstance fieldAcceptorInstance : fieldAcceptors) {
                final Stats stats = statsByField.get(fieldAcceptorInstance);
                final StructuredStat structuredStat = structuredStatsByField.get(fieldAcceptorInstance);

                if (stats != null && !stats.isEmpty() || structuredStat != null && !structuredStat.isEmpty()) {
                    requestsConsumer.accept(new CollapsedAggregatesUpdateRequest(
                            key,
                            fieldAcceptorInstance,
                            stats,
                            structuredStat,
                            averageDay));
                }
            }
        }
    }

    @Nonnull
    private Instant getAverageDay() {
        return earliestTimestamp.plus(Duration.between(earliestTimestamp, latestTimestamp).dividedBy(2)).truncatedTo(ChronoUnit.DAYS);
    }
}
