import argparse
import logging
import os
from collections import namedtuple
from datetime import timedelta, datetime

import requests

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger()

TypeDeletionTraits = namedtuple('TypeDeletionTraits', ('prefix', 'services', 'values', 'nots'))

DAY_MILLISECONDS = timedelta(days=1).total_seconds() * 1000
NOW_MILLISECONDS = int(os.getenv("NOW_MILLISECONDS", datetime.now().timestamp() * 1000))


def day_gap_to_query(day_gap):
    ms = int(NOW_MILLISECONDS + DAY_MILLISECONDS * day_gap)
    return f"[0+TO+{ms}]"


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--host', type=str, required=True)
    parser.add_argument('--COLLAPSED_AGGRS_DAY_TO_DELETE', type=int, required=True)
    parser.add_argument('--PASSPORT_UID_COLLAPSED_AGGRS_DAY_TO_DELETE', type=int, required=True)

    args = parser.parse_args()

    traits = (
        TypeDeletionTraits(
            1,
            ("so_fraud_data", "so_fraud_testing"),
            (
                ("type", "COLLAPSED_AGGRS"),
                ("txn_day", day_gap_to_query(args.COLLAPSED_AGGRS_DAY_TO_DELETE)),
            ),
            ()
        ),
        TypeDeletionTraits(
            1,
            ("so_fraud_passport",),
            (
                ("type", "COLLAPSED_AGGRS"),
                ("txn_day", day_gap_to_query(args.COLLAPSED_AGGRS_DAY_TO_DELETE)),
            ),
            (
                ("key", "uid"),
            )
        ),
        TypeDeletionTraits(
            1,
            ("so_fraud_passport",),
            (
                ("type", "COLLAPSED_AGGRS"),
                ("txn_day", day_gap_to_query(args.PASSPORT_UID_COLLAPSED_AGGRS_DAY_TO_DELETE)),
                ("key", "uid"),
            ),
            ()
        ),
    )

    for trait in traits:
        for service in trait.services:
            text = "+AND+".join(f"{key}:{value}" for key, value in trait.values)
            if trait.nots:
                text += "".join(f"+AND+NOT({key}:{value})" for key, value in trait.nots)
            url = f"{args.host}/delete?prefix={trait.prefix}&service={service}&text={text}"
            logger.info(url)
            response = requests.get(url)
            response.raise_for_status()


if __name__ == "__main__":
    main()
