local ValueStorage = require('value_storage');
local luajava = require('luajava');
local Pattern = luajava.bindClass("java.util.regex.Pattern");

local pattern = {};

local LuaPattern = {
    compile = function(self, p)
        local o = {}
        setmetatable(o, self)
        self.__index = self;
        o.__pattern = Pattern:compile(p, Pattern.UNICODE_CASE);
        return o;
    end,
    matches = function(self, stack)
        return self.__pattern:matcher(stack):matches();
    end,
    contains = function(self, stack)
        return self.__pattern:matcher(stack):find();
    end
};

---@overload fun(p:string, flags:number):LuaPattern
function pattern.compile(p)
    local re = {};
    setmetatable(re, LuaPattern);
    return LuaPattern:compile(p);
end

local NEEDLES = ValueStorage:new();

function pattern.matches(needle, stack)
    local re = NEEDLES:get_or_compute(needle, function()
        return pattern.compile(needle);
    end);
    return re:matches(stack);
end

function pattern.contains(needle, stack)
    local re = NEEDLES:get_or_compute(needle, function()
        return pattern.compile(needle);
    end);
    return re:contains(stack);
end

----tests-start----
local function __tests__()
    local re = pattern.compile(".*[TtТт]\\s*[EeЕе]\\s*[KkКк]\\s*[CcСс]\\s*[TtТт].*");
    assert(re:matches("текст"));
    assert(re:matches("tekct"));
    assert(re:matches("  tekct "));
    assert(not re:matches("qwerty"));

    re = pattern.compile("(?i)текст");
    assert(re:matches("ТеКсТ"));

    assert(pattern.matches("(?i)текст", "ТеКсТ"));

    assert(pattern.contains("(?i)украин|войн|войск|солдат|путин|митинг|шествие", "Слава Украине! Героям - слава"))
end
__tests__()
----tests-end----

return pattern;

