import argparse
import json
import requests
from tqdm import tqdm
import pathlib


def read_data(path: pathlib.Path):
    with open(path) as f:
        for line in f:
            yield json.loads(line)


def make_document(data, key):
    value = data.get(key)
    storage_service = data.get("storage_service")
    if not value or not storage_service:
        return None, None

    key_value = f"{key}_{value}"
    channel = data.get("channel")
    sub_channel = data.get("sub_channel")
    channel_uri = data.get("channel_uri")

    txn_timestamp = data["txn_timestamp"]
    txn_status_timestamp = data["txn_status_timestamp"]

    return storage_service, {
        "id": f"txn_{channel_uri}_{key_value}_counters",
        "channel": data.get("channel"),
        "sub_channel": channel,
        "channel_uri": sub_channel,
        "key_value": key_value,
        "key": key,
        "value": value,
        key: value,
        "type": "COUNTERS",
        "txn_timestamp": {
            "min": [
                txn_timestamp,
                {"get": ["txn_timestamp"]}
            ]
        },
        "txn_status_timestamp": {
            "max": [
                txn_status_timestamp,
                {"get": ["txn_status_timestamp"]}
            ]
        }
    }


def generate_documents(uids_file, card_ids_file):
    for (path, key) in (
        (uids_file, "uid"),
        (card_ids_file, "card_id"),
    ):
        if not path:
            continue

        for data in read_data(path):
            storage_service, document = make_document(data, key)
            if not document or not storage_service:
                continue
            yield storage_service, document


def send_batch(session: requests.Session,
               batch: list[dict],
               storage_service,
               host: str):
    response = session.post(
        f"{host}/update?prefix={0}&service={storage_service}",
        json={
            "AddIfNotExists": True,
            "prefix": 0,
            "docs": batch,
        },
    )

    response.raise_for_status()


parser = argparse.ArgumentParser()
parser.add_argument("--host", type=str, required=True)
parser.add_argument("--uids_file", type=pathlib.Path)
parser.add_argument("--card_ids_file", type=pathlib.Path)
parser.add_argument("--batch_size", type=int, default=1000)


def main():
    args = parser.parse_args()

    batches = {}

    with requests.Session() as session:
        for storage_service, document in tqdm(
            generate_documents(args.uids_file, args.card_ids_file),
            desc="Progress",
            unit="row",
        ):
            batch = batches.setdefault(storage_service, [])

            batch.append(document)

            if len(batch) >= args.batch_size:
                send_batch(session, batch, storage_service, args.host)
                batches[storage_service] = []

    for storage_service, batch in batches:
        if len(batch) > 0:
            send_batch(session, batch, storage_service, args.host)


if __name__ == "__main__":
    main()
