import argparse
from datetime import datetime
import requests
from tqdm import tqdm
import yt.wrapper
import typing

PREFIX = 0
SERVICE = "so_fraud_login_id"


@yt.wrapper.yt_dataclass
class VerificationLevelTuple:
    device_id: typing.Optional[str]
    login_id: typing.Optional[str]
    uid: typing.Optional[str]

    def make_document(self, verification_level: str, date: datetime) -> str:
        now_ms = int(date.timestamp()) * 1000
        return {
            "id": f"txn_verification_level_{self.uid}_{self.login_id}_{self.device_id}_{verification_level}_{date.ctime()}",
            "type": "VERIFICATION_LEVEL",
            "uid": self.uid,
            "txn_timestamp": now_ms,
            "txn_status_timestamp": now_ms,
            "txn_status": "OK",
            "device_id": self.device_id,
            "login_id": self.login_id,
            "afs_verification_level": verification_level,
            "service_id": "login",
        }


def send_batch(session: requests.Session, batch: list[dict], host: str):
    response = session.post(
        f"{host}/update?prefix={PREFIX}&service={SERVICE}",
        json={
            "AddIfNotExists": True,
            "prefix": PREFIX,
            "docs": batch,
        },
    )

    response.raise_for_status()


parser = argparse.ArgumentParser()
parser.add_argument("--host", type=str, required=True)
parser.add_argument("--proxy", type=str, default="hahn")
parser.add_argument("--src", type=yt.wrapper.TablePath, required=True)
parser.add_argument("--batch-size", type=int, default=100)
parser.add_argument("--verification-level", type=str, required=True)


def main():
    args = parser.parse_args()

    now = datetime.now()

    client = yt.wrapper.YtClient(proxy=args.proxy)

    row_count = client.row_count(args.src)

    batch = []

    with requests.Session() as session:
        verification_tuple: VerificationLevelTuple
        for verification_tuple in tqdm(
            client.read_table_structured(args.src, VerificationLevelTuple),
            desc="Progress",
            total=row_count,
            unit="row",
        ):
            batch.append(verification_tuple.make_document(args.verification_level, now))

            if len(batch) >= args.batch_size:
                send_batch(session, batch, args.host)
                batch = []

        if len(batch) > 0:
            send_batch(session, batch, args.host)


if __name__ == "__main__":
    main()
