package ru.yandex.mail.so.cretur;

import java.io.IOException;
import java.util.Arrays;

import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.pg.SqlQuery;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.proxy.StaticEntitySendingCallback;
import ru.yandex.http.tskv.TskvProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.PayloadTooLargeException;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.util.ip.Cidr;

public class SetOrganizationIpWhitelistHandler
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final SqlQuery SET_ORGANIZATION_IP_WHITELIST =
        new SqlQuery(
            "set-organization-ip-whitelist.sql",
            SetOrganizationIpWhitelistHandler.class);

    private final Cretur server;

    public SetOrganizationIpWhitelistHandler(final Cretur server) {
        this.server = server;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        throw new BadRequestException("Payload expected");
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession basicSession =
            new BasicProxySession(server, exchange, context);
        long orgId = basicSession.params().getLong("org_id");
        String[] cidrs;
        try {
            JsonList whitelist = payload.get("ip_whitelist").asList();
            int size = whitelist.size();
            int ipWhitelistMaxSize = server.config().ipWhitelistMaxSize();
            if (size > ipWhitelistMaxSize) {
                throw new PayloadTooLargeException(
                    "IP whitelist max size is " + ipWhitelistMaxSize
                    + ", but " + size + " IPs received");
            }

            cidrs = new String[size];
            for (int i = 0; i < size; ++i) {
                String cidr = whitelist.get(i).asString();
                Cidr.fromString(cidr);
                cidrs[i] = cidr;
            }
        } catch (JsonException | IOException e) {
            throw new BadRequestException("Malformed request", e);
        }

        TskvProxySession session =
            new TskvProxySession(
                basicSession,
                server,
                this,
                "Setting organization ip whitelist to "
                + Arrays.toString(cidrs),
                TskvFields.ORG_ID,
                Long.toString(orgId));

        if (cidrs.length == 0) {
            DeleteOrganizationIpWhitelistHandler.deleteOrganizationIpWhitelist(
                server.pgClient(),
                orgId,
                session);
        } else {
            Tuple tuple = Tuple.tuple();
            tuple.addLong(orgId);
            tuple.addArrayOfString(cidrs);

            server.pgClient().executeOnMaster(
                SET_ORGANIZATION_IP_WHITELIST,
                tuple,
                session.listener(),
                new StaticEntitySendingCallback<>(
                    session,
                    rowSet ->
                        "Rows affected by update: " + rowSet.rowCount()));
        }
    }
}

