package ru.yandex.mail.so.cwacf;

import java.io.IOException;
import java.io.StringReader;
import java.util.List;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.rpc.JsonRpcErrorCode;
import ru.yandex.json.rpc.JsonRpcException;
import ru.yandex.json.rpc.JsonRpcRequest;
import ru.yandex.json.rpc.JsonRpcResponse;
import ru.yandex.json.writer.JsonType;
import ru.yandex.stater.BooleanStater;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.xml.xpath.XPathContentHandler;
import ru.yandex.xml.xpath.XPathLeafHandler;

public class Cwacf
    extends HttpProxy<ImmutableCwacfConfig>
    implements HttpAsyncRequestHandler<JsonObject>
{
    private static final Object NULL_REPLACEMENT = new Object();

    private final ImmutableHttpHostConfig upstreamConfig;
    private final AsyncClient upstreamClient;
    private final TimeFrameQueue<JsonRpcErrorCode> errorCodes;
    private final TimeFrameQueue<Object> resolutions;

    public Cwacf(final ImmutableCwacfConfig config) throws IOException {
        super(config);
        upstreamConfig = config.upstreamConfig();
        upstreamClient = client("Upstream", upstreamConfig);
        errorCodes = new TimeFrameQueue<>(config.metricsTimeFrame());
        resolutions = new TimeFrameQueue<>(config.metricsTimeFrame());
        register(
            new Pattern<>("", true),
            this,
            RequestHandlerMapper.POST);
        registerStater(
            new PassiveStaterAdapter<>(
                errorCodes,
                new EnumStaterFactory<>(
                    code -> "error-code-" + code + "_ammm",
                    JsonRpcErrorCode.values())));
        registerStater(
            new BooleanStater(resolutions, "checkform-resolutions-"));
    }

    public ImmutableHttpHostConfig upstreamConfig() {
        return upstreamConfig;
    }

    public AsyncClient upstreamClient() {
        return upstreamClient;
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        throw new BadRequestException("Request body expected");
    }

    public void sendResponse(
        final ProxySession session,
        final JsonRpcResponse response)
    {
        sendResponse(session, JsonType.NORMAL, response);
    }

    public void sendResponse(
        final ProxySession session,
        final JsonType jsonType,
        final JsonRpcResponse response)
    {
        JsonRpcException error = response.error();
        if (error != null) {
            JsonRpcErrorCode code = error.code();
            errorCodes.accept(code);
            session.connection()
                .setSessionInfo("json_rpc_error", code.toString());
            StringBuilder sb = new StringBuilder("Request processing failed");
            JsonObject data = error.data();
            if (data != null) {
                sb.append(':');
                sb.append(' ');
                JsonType.HUMAN_READABLE.toStringBuilder(sb, data);
            }
            session.logger().log(
                Level.WARNING,
                new String(sb),
                error);
        }
        session.response(
            HttpStatus.SC_OK,
            new NStringEntity(
                jsonType.toString(response),
                ContentType.APPLICATION_JSON.withCharset(
                    session.acceptedCharset())));
    }

    public void accountResolution(final Object resolution) {
        resolutions.accept(resolution);
    }

    @Override
    public void handle(
        final JsonObject requestObject,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(this, exchange, context);
        JsonRpcRequest request;
        try {
            request = new JsonRpcRequest(requestObject);
        } catch (JsonException e) {
            sendResponse(
                session,
                new JsonRpcResponse(
                    null,
                    new JsonRpcException(
                        JsonRpcErrorCode.INVALID_REQUEST,
                        "Failed to parse rpc request",
                        requestObject,
                        e)));
            return;
        }
        try {
            CwacfRequestContext requestContext =
                new CwacfRequestContext(this, session, request);
            AsyncClient client = upstreamClient.adjust(session.context());
            client.execute(
                upstreamConfig.host(),
                new BasicAsyncRequestProducerGenerator(
                    requestContext.query(),
                    requestContext.body(),
                    ContentType.APPLICATION_JSON.withCharset(
                        upstreamConfig.requestCharset())),
                AsyncStringConsumerFactory.OK,
                session.listener().createContextGeneratorFor(client),
                new Callback(requestContext));
        } catch (JsonRpcException e) {
            sendResponse(session, new JsonRpcResponse(request, e));
        } catch (HttpException e) {
            sendResponse(
                session,
                new JsonRpcResponse(
                    request,
                    new JsonRpcException(
                        JsonRpcErrorCode.INVALID_REQUEST,
                        "Failed to construct request context",
                        requestObject,
                        e)));
        }
    }

    private static class Callback implements FutureCallback<String> {
        private final CwacfRequestContext requestContext;

        Callback(final CwacfRequestContext requestContext) {
            this.requestContext = requestContext;
        }

        @Override
        public void failed(final Exception e) {
            Cwacf cwacf = requestContext.cwacf();
            cwacf.accountResolution(NULL_REPLACEMENT);
            cwacf.sendResponse(
                requestContext.session(),
                requestContext.jsonType(),
                new JsonRpcResponse(
                    requestContext.request(),
                    new JsonRpcException(
                        JsonRpcErrorCode.INTERNAL_ERROR,
                        "CheckForm failed",
                        e)));
        }

        @Override
        public void cancelled() {
            requestContext.session().logger().warning("Request cancelled");
        }

        @Override
        public void completed(final String response) {
            requestContext.session().logger().fine(
                "Checkform response: " + response);
            try {
                CheckformResponseHandler handler =
                    new CheckformResponseHandler();
                XMLReader reader = XMLReaderFactory.createXMLReader();
                reader.setContentHandler(new XPathContentHandler(handler));
                reader.parse(new InputSource(new StringReader(response)));
                boolean spam;
                String resolution;
                if ("1".equals(handler.resolution)) {
                    spam = true;
                    resolution = "SPAM";
                } else if ("0".equals(handler.resolution)) {
                    spam = false;
                    resolution = "HAM";
                } else {
                    spam = false;
                    resolution = "UNKNOWN";
                }
                Cwacf cwacf = requestContext.cwacf();
                cwacf.accountResolution(spam);
                requestContext.session().connection().setSessionInfo(
                    "resolution",
                    resolution);

                JsonMap verdict = new JsonMap(BasicContainerFactory.INSTANCE);
                verdict.put("name", new JsonString("text_so_is_spam"));
                verdict.put("key", new JsonString(requestContext.key()));
                verdict.put("entity", new JsonString(requestContext.type()));
                verdict.put("source", new JsonString("checkform"));
                verdict.put("subsource", new JsonString("checkform"));
                verdict.put("value", JsonBoolean.valueOf(spam));
                JsonList result = new JsonList(BasicContainerFactory.INSTANCE);
                result.add(verdict);
                cwacf.sendResponse(
                    requestContext.session(),
                    requestContext.jsonType(),
                    new JsonRpcResponse(
                        requestContext.request(),
                        result));
            } catch (IOException | SAXException e) {
                requestContext.cwacf().sendResponse(
                    requestContext.session(),
                    new JsonRpcResponse(
                        requestContext.request(),
                        new JsonRpcException(
                            JsonRpcErrorCode.PARSE_ERROR,
                            "Response parse failed: <" + response + '>',
                            e)));
            }
        }
    }

    private static class CheckformResponseHandler implements XPathLeafHandler {
        private String resolution = null;

        @Override
        public void handle(final List<String> path, final String text) {
            if (path.size() == 1 && path.get(0).equals("spam")) {
                resolution = text;
            }
        }
    }
}

