package ru.yandex.mail.so.jrbld;

import java.net.InetAddress;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.util.string.StringUtils;

public class ChainedIpChecker implements AsyncIpChecker {
    private final String name;
    private final IpSetCheckerAdapter[] checkers;
    private final AsyncIpChecker[] asyncCheckers;
    private final String noneMatched;
    private final String subChecker;

    public ChainedIpChecker(
        final String name,
        final List<AsyncIpChecker> checkers)
    {
        this.name = name;
        int firstAsync = 0;
        while (firstAsync < checkers.size()) {
            if (checkers.get(firstAsync) instanceof IpSetCheckerAdapter) {
                ++firstAsync;
            } else {
                break;
            }
        }
        this.checkers = new IpSetCheckerAdapter[firstAsync];
        for (int i = 0; i < firstAsync; ++i) {
            this.checkers[i] = (IpSetCheckerAdapter) checkers.get(i);
        }
        asyncCheckers = new AsyncIpChecker[checkers.size() - firstAsync];
        for (int i = firstAsync; i < checkers.size(); ++i) {
            asyncCheckers[i - firstAsync] = checkers.get(i);
        }
        String forChecker = StringUtils.concat("For checker '", name, "' ");
        noneMatched =
            StringUtils.concat(forChecker, "none of subcheckers matched ip <");
        subChecker = StringUtils.concat(forChecker, "sub-checker '");
    }

    @Override
    public void check(
        final IpCheckRequest request,
        final FutureCallback<? super Integer> callback)
    {
        InetAddress ip = request.ip();
        for (IpSetCheckerAdapter checker: checkers) {
            if (checker.test(ip)) {
                callback.completed(checker.value());
                return;
            }
        }
        check(0, request, callback);
    }

    private void check(
        final int index,
        final IpCheckRequest request,
        final FutureCallback<? super Integer> callback)
    {
        if (index >= asyncCheckers.length) {
            request.session().logger().info(
                StringUtils.concat(
                    noneMatched,
                    request.ipString(),
                    '>'));
            callback.completed(ZERO);
        } else if (request.cancelled()) {
            request.session().logger()
                .info("Request cancelled, chain execution stopped");
            callback.completed(ZERO);
        } else {
            asyncCheckers[index].check(
                request,
                new Callback(callback, request, index));
        }
    }

    @Override
    public String toString() {
        return name;
    }

    private class Callback extends FilterFutureCallback<Integer> {
        private final IpCheckRequest request;
        private final int index;

        Callback(
            final FutureCallback<? super Integer> callback,
            final IpCheckRequest request,
            final int index)
        {
            super(callback);
            this.request = request;
            this.index = index;
        }

        @Override
        public void failed(final Exception e) {
            request.session().logger().log(
                Level.WARNING,
                StringUtils.concat(
                    subChecker,
                    asyncCheckers[index].toString(),
                    "' failed for ip <",
                    request.ipString(),
                    '>'),
                e);
            super.failed(e);
        }

        @Override
        public void completed(final Integer result) {
            if (ZERO.equals(result)) {
                check(index + 1, request, callback);
            } else {
                Logger logger = request.session().logger();
                if (logger.isLoggable(Level.FINE)) {
                    request.session().logger().fine(
                        StringUtils.concat(
                            subChecker,
                            asyncCheckers[index].toString(),
                            "' matched ip <",
                            request.ipString(),
                            '>'));
                }
                callback.completed(result);
            }
        }
    }
}

