package ru.yandex.mail.so.jrbld;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.http.server.async.DelegatedHttpAsyncRequestHandler;
import ru.yandex.jniwrapper.ImmutableJniWrapperConfig;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.mail.so.jrbld.config.ImmutableCidrListSourceConfig;
import ru.yandex.mail.so.jrbld.config.ImmutableIpCheckerConfig;
import ru.yandex.mail.so.jrbld.config.ImmutableIpListSourceConfig;
import ru.yandex.mail.so.jrbld.config.ImmutableIpRangeListSourceConfig;
import ru.yandex.mail.so.jrbld.config.ImmutableJrbldConfig;
import ru.yandex.mail.so.jrbld.config.ImmutableLuceneSourceConfig;
import ru.yandex.mail.so.jrbld.config.IpCheckerValueConfig;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.stater.BooleanStater;
import ru.yandex.util.ip.ReloadableIpCheckerFactory;
import ru.yandex.util.string.StringUtils;

public class Jrbld extends UniversalSearchProxy<ImmutableJrbldConfig> {
    private static final String IP_CHECKER = "ip-checker-";

    private final Map<String, IpInfoProvider> infoProviders;
    private final Map<String, AsyncIpChecker> checkers;

    public Jrbld(final ImmutableJrbldConfig config)
        throws IOException, JniWrapperException
    {
        super(config);
        try (GenericAutoCloseableHolder<
                IOException,
                GenericAutoCloseableChain<IOException>> holder =
                new GenericAutoCloseableHolder<>(closeChain))
        {
            Map<String, ImmutableJniWrapperConfig> jniInfoSourceConfigs =
                config.jniInfoSourceConfigs();
            infoProviders = new HashMap<>(jniInfoSourceConfigs.size() << 1);
            for (Map.Entry<String, ImmutableJniWrapperConfig> entry
                : jniInfoSourceConfigs.entrySet())
            {
                String name = entry.getKey();
                JniWrapper jniwrapper =
                    JniWrapper.create(
                        entry.getValue(),
                        new ThreadFactoryConfig(
                            config.name() + '-' + name + '-')
                            .group(getThreadGroup())
                            .daemon(true));
                closeChain.add(
                    new GenericNonThrowingCloseableAdapter<>(jniwrapper));
                infoProviders.put(name, new JniIpInfoProvider(jniwrapper));
            }

            Map<String, ImmutableLuceneSourceConfig>
                luceneSourceConfigs = config.luceneSourceConfigs();
            Map<String, ImmutableIpListSourceConfig>
                ipListSourceConfigs = config.ipListSourceConfigs();
            Map<String, ImmutableIpRangeListSourceConfig>
                ipRangeListSourceConfigs =
                    config.ipRangeListSourceConfigs();
            Map<String, ImmutableCidrListSourceConfig>
                cidrListSourceConfigs =
                    config.cidrListSourceConfigs();
            Map<String, ImmutableIpCheckerConfig> ipCheckerConfigs =
                config.ipCheckerConfigs();
            checkers = new HashMap<>(
                (luceneSourceConfigs.size()
                + ipListSourceConfigs.size()
                + ipRangeListSourceConfigs.size()
                + cidrListSourceConfigs.size()
                + ipCheckerConfigs.size()) << 1);

            for (Map.Entry<String, ImmutableLuceneSourceConfig> entry
                : luceneSourceConfigs.entrySet())
            {
                String name = entry.getKey();
                AsyncIpChecker checker =
                    new LuceneIpChecker(this, entry.getValue());
                if (config.enableCheckersStaters()) {
                    TimeFrameQueue<Object> stater =
                        new TimeFrameQueue<>(config.metricsTimeFrame());
                    registerStater(
                        new BooleanStater(
                            stater,
                            StringUtils.concat(IP_CHECKER, name, '-')));
                    checker = new StatingIpChecker(checker, stater);
                }
                checkers.put(name, checker);
            }

            List<ReloadableAsyncIpChecker> reloadableCheckers =
                new ArrayList<>(
                    ipListSourceConfigs.size()
                    + ipRangeListSourceConfigs.size()
                    + cidrListSourceConfigs.size());
            addCheckers(config, ipListSourceConfigs, reloadableCheckers);
            addCheckers(config, ipRangeListSourceConfigs, reloadableCheckers);
            addCheckers(config, cidrListSourceConfigs, reloadableCheckers);

            for (Map.Entry<String, ImmutableIpCheckerConfig> entry
                : ipCheckerConfigs.entrySet())
            {
                List<AsyncIpChecker> chainedCheckers = new ArrayList<>();
                for (String subChecker: entry.getValue().sources()) {
                    chainedCheckers.add(checkers.get(subChecker));
                }
                String name = entry.getKey();
                AsyncIpChecker checker =
                    new ChainedIpChecker(name, chainedCheckers);
                if (config.enableCheckersStaters()) {
                    TimeFrameQueue<Object> stater =
                        new TimeFrameQueue<>(config.metricsTimeFrame());
                    registerStater(
                        new BooleanStater(
                            stater,
                            StringUtils.concat(IP_CHECKER, name, '-')));
                    checker = new StatingIpChecker(checker, stater);
                }
                checkers.put(name, checker);
            }

            register(
                new Pattern<>("/check", false),
                new CheckIpHandler(this, infoProviders, checkers));
            register(
                new Pattern<>("/check-all", false),
                new CheckIpsHandler(this, infoProviders, checkers));
            register(
                new Pattern<>("/reloadlists", false),
                new DelegatedHttpAsyncRequestHandler<>(
                    new ReloadListsHandler(reloadableCheckers),
                    this));

            holder.release();
        }
    }

    private <T extends IpCheckerValueConfig & ReloadableIpCheckerFactory>
        void addCheckers(
            final ImmutableJrbldConfig jrbldConfig,
            final Map<String, T> configs,
            final List<ReloadableAsyncIpChecker> reloadableCheckers)
        throws IOException
    {
        for (Map.Entry<String, T> entry: configs.entrySet())
        {
            String name = entry.getKey();
            T config = entry.getValue();
            ReloadableAsyncIpChecker reloadableChecker =
                new IpSetCheckerAdapter(
                    config.value(),
                    config.createIpChecker());
            reloadableCheckers.add(reloadableChecker);
            AsyncIpChecker checker = reloadableChecker;
            if (jrbldConfig.enableCheckersStaters()) {
                TimeFrameQueue<Object> stater =
                    new TimeFrameQueue<>(jrbldConfig.metricsTimeFrame());
                registerStater(
                    new BooleanStater(
                        stater,
                        StringUtils.concat(IP_CHECKER, name, '-')));
                checker = new StatingIpChecker(checker, stater);
            }
            checkers.put(name, checker);
        }
    }
}

