package ru.yandex.mail.so.jrbld;

import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.cache.async.AsyncCache;
import ru.yandex.cache.async.AsyncCacheResultCallback;
import ru.yandex.cache.async.AsyncCacheResultType;
import ru.yandex.cache.async.AsyncLoader;
import ru.yandex.cache.async.ConcurrentLinkedHashMapAsyncStorage;
import ru.yandex.cache.async.ConcurrentLinkedHashMapAsyncStorageStater;
import ru.yandex.cache.async.ConstAsyncCacheTtlCalculator;
import ru.yandex.cache.async.FixedWeigher;
import ru.yandex.cache.async.StringWeigher;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.mail.so.jrbld.config.ImmutableLuceneSourceConfig;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.search.result.SearchResult;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.timesource.TimeSource;

public class LuceneIpChecker implements AsyncIpChecker {
    private static final long INT_MASK = 0xffffffffL;
    private static final long CACHE_CAPACITY = 32L << 20;
    private static final long TTL = 60000L;

    private final String name;
    private final TimeFrameQueue<AsyncCacheResultType> luceneCacheHitType;
    private final AsyncCache<
        String,
        Integer,
        IpCheckRequest,
        RuntimeException>
        cache;

    public LuceneIpChecker(
        final Jrbld jrbld,
        final ImmutableLuceneSourceConfig config)
    {
        name = config.query().replaceAll("[^a-zA-Z0-9]+", "-");
        ConcurrentLinkedHashMapAsyncStorage<String, Integer> storage =
            new ConcurrentLinkedHashMapAsyncStorage<>(
                CACHE_CAPACITY,
                Runtime.getRuntime().availableProcessors(),
                StringWeigher.INSTANCE,
                FixedWeigher.BOXED_NUMBER_WEIGHER);
        luceneCacheHitType =
            new TimeFrameQueue<>(jrbld.config().metricsTimeFrame());
        jrbld.registerStater(
            new PassiveStaterAdapter<>(
                luceneCacheHitType,
                new EnumStaterFactory<>(
                    result -> result + "_ammm",
                    AsyncCacheResultType.values(),
                    name + "-cache-hit-type-",
                    "lucene-cache",
                    "lucene cache hit type",
                    null,
                    1)));
        jrbld.registerStater(
            new ConcurrentLinkedHashMapAsyncStorageStater(
                storage,
                true,
                name + '-',
                "lucene-cache",
                null));
        cache = new AsyncCache<>(
            storage,
            new ConstAsyncCacheTtlCalculator(TTL),
            new Loader(jrbld, config));
    }

    @Override
    public void check(
        final IpCheckRequest request,
        final FutureCallback<? super Integer> callback)
    {
        cache.get(
            request.ipString(),
            request,
            new AsyncCacheResultCallback<>(
                callback,
                request.session().logger().addPrefix(name),
                luceneCacheHitType));
    }

    @Override
    public String toString() {
        return name;
    }

    private static class RequestContext
        implements UniversalSearchProxyRequestContext
    {
        private final User user;
        private final AsyncClient client;
        private final Logger logger;

        RequestContext(
            final User user,
            final AsyncClient client,
            final Logger logger)
        {
            this.user = user;
            this.client = client;
            this.logger = logger;
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AsyncClient client() {
            return client;
        }

        @Override
        public Logger logger() {
            return logger;
        }

        @Override
        public long lagTolerance() {
            return Long.MAX_VALUE;
        }
    }

    private static class Callback
        extends AbstractFilterFutureCallback<SearchResult, Integer>
    {
        private final Integer value;

        Callback(
            final FutureCallback<? super Integer> callback,
            final Integer value)
        {
            super(callback);
            this.value = value;
        }

        @Override
        public void completed(final SearchResult result) {
            Integer value;
            if (result.hitsCount() > 0L) {
                value = this.value;
            } else {
                value = AsyncIpChecker.ZERO;
            }
            callback.completed(value);
        }
    }

    private static class Loader
        implements AsyncLoader<String, Integer, IpCheckRequest>
    {
        private final Jrbld jrbld;
        private final Integer value;
        private final String query;
        private final String service;
        private final long maxPrefix;
        private final Long failoverDelay;

        Loader(
            final Jrbld jrbld,
            final ImmutableLuceneSourceConfig config)
        {
            this.jrbld = jrbld;
            value = config.value();
            query = StringUtils.concat("\" AND (", config.query(), ')');
            service = config.service();
            maxPrefix = config.maxPrefix();
            failoverDelay = config.failoverDelay();
        }

        @Override
        public void load(
            final String ipString,
            final IpCheckRequest request,
            final FutureCallback<? super Integer> callback)
        {
            try {
                long prefix = (request.ip().hashCode() & INT_MASK) % maxPrefix;
                QueryConstructor query = new QueryConstructor(
                    "/search?json-type=dollar&service=" + service
                    + "&length=0&collector=passthrough%280%29&prefix=" + prefix
                    + "&postfilter=rbl_expire_timestamp+%3e%3d+"
                    + TimeSource.INSTANCE.currentTimeMillis());
                query.append(
                    "text",
                    StringUtils.concat(
                        "rbl_ip:\"",
                        ipString,
                        this.query));
                ProxySession session = request.session();
                AsyncClient client =
                    jrbld.searchClient().adjust(session.context());
                jrbld.sequentialRequest(
                    session,
                    new RequestContext(
                        new User(service, new LongPrefix(prefix)),
                        client,
                        session.logger()),
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    failoverDelay,
                    true,
                    SearchResultConsumerFactory.OK,
                    session.listener().createContextGeneratorFor(client),
                    new Callback(callback, value));
            } catch (BadRequestException e) {
                callback.failed(e);
            }
        }
    }
}

