import asyncio
import logging

import configargparse
import yt.wrapper as yt
from aiohttp import web
from aiohttp.web_request import Request
from aiohttp.web_response import Response

from .bot import Bot
from .models import Engine
from .reports_producer import ReportsProducer
from .screenshooter import ScreenShooter
from .staff_validator import StaffValidator
from .tvm import TvmClient


async def ping(request: Request):
    return Response(status=200)


def main():
    parser = configargparse.ArgumentParser()
    parser.add_argument("--telegram-token", type=str, env_var="TELEGRAM_TOKEN", help="Telegram bot token")
    parser.add_argument("--oauth-token", type=str, env_var="OAUTH_TOKEN", help="Oauth token")
    parser.add_argument("--port", type=int, help="port", required=True)
    parser.add_argument("--tvm-self-id", type=int, required=True)
    parser.add_argument("--tvm-secret", type=str, env_var="TVM_SECRET")
    parser.add_argument("--bb-id", type=int, required=True)
    parser.add_argument("--allowed-consumers", required=True)
    parser.add_argument("--db-path", type=str, required=True)
    parser.add_argument("--compl-host", type=str, required=True)
    parser.add_argument("--reports-dir", type=str, required=True)
    parser.add_argument("--falses-table", type=str, required=True)
    parser.add_argument("--screenshots-dir", type=str, default='/data')
    parser.add_argument("--chrome-bin", type=str, required=True)
    parser.add_argument("--chrome-driver", type=str, required=True)
    parser.add_argument("--render-timeout", type=int, required=True)

    args = parser.parse_args()

    logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')

    staff_user_validator = StaffValidator(args.oauth_token)
    tvm_client = TvmClient(self_id=args.tvm_self_id, secret=args.tvm_secret, bb_id=args.bb_id,
                           allowed_consumers=args.allowed_consumers.split())

    engine = Engine(args.db_path)

    yt.update_config({"proxy": {"url": "hahn"}, "token": args.oauth_token})

    reports_producer = ReportsProducer(args.reports_dir, engine)

    screenshooter = ScreenShooter(args.chrome_bin, args.chrome_driver, args.screenshots_dir, args.render_timeout)

    bot = Bot(args.telegram_token, staff_user_validator, tvm_client, engine, reports_producer, screenshooter,
              args.compl_host, args.falses_table)

    app = web.Application()
    app.router.add_get('/ping', ping)

    runner = web.AppRunner(app)
    loop = asyncio.get_event_loop()

    loop.run_until_complete(runner.setup())

    loop.create_task(reports_producer.fetch(60))
    loop.create_task(bot.broadcast_group())
    loop.create_task(web.TCPSite(runner, 'localhost', args.port).start())

    loop.run_until_complete(bot.bot.loop())
    logging.info("Stopped")


if __name__ == "__main__":
    main()
