import asyncio
import logging
import os
import traceback
import typing
from datetime import datetime

import aiotg
import selenium.common.exceptions
import ujson
import yt.wrapper as yt

from mail.so.libs.python.cast import to_ulonglong
from .models import Chat, Engine, Signature, Choice
from .reports_group import ReportsSignatureGroup
from .reports_producer import ReportsProducer
from .screenshooter import ScreenShooter
from .tvm import TvmClient
from .user_validator import Validator
from .view import View, FalseButton, LaterButton, Button

logger = logging.getLogger("bot")


class Bot:

    def __init__(self,
                 telegram_token: str,
                 user_validator: Validator,
                 tvm_client: TvmClient,
                 engine: Engine,
                 reports_producer: ReportsProducer,
                 screenshooter: ScreenShooter,
                 comp_reputation_host: str,
                 falses_table: str,
                 refresh_time: int = 5):
        self.bot = aiotg.Bot(api_token=telegram_token, api_timeout=-1, default_in_groups=True)

        self.tvm_client = tvm_client
        self.user_validator = user_validator
        self.bot.add_command(r"/help", self.usage)
        self.bot.add_command(r"/start", self.start)
        self.bot.add_command(r"/chats", self.get_chats)
        self.bot.add_command(r"/w", self.set_weight)
        self.bot.add_command(r"/d", self.set_duration)
        self.bot.add_command(r"/c", self.send_current_group)
        self.bot.add_command(r"/l", self.get_reports_count)
        self.bot.add_command(r"/test", self.test)
        self.bot.add_callback(r"button-(.+)", self.proceed_callback)
        self.bot.default(self.error)
        self.engine = engine
        self.reports_producer = reports_producer
        self.screenshooter = screenshooter
        self.reports_groups: typing.Dict[Signature, ReportsSignatureGroup] = {}
        self.current_group: typing.Optional[ReportsSignatureGroup] = None
        self.reports_iterator = None
        self.refresh_time = refresh_time
        self.falses_table = yt.TablePath(falses_table, append=True)

        if not yt.exists(self.falses_table):
            schema = [
                {"name": "domain_hash", "type": "uint64"},
                {"name": "shingle", "type": "uint64"},
                {"name": "shingle_type", "type": "uint32"},
                {"name": "date", "type": "datetime"},
            ]
            records = []
            for choice in reports_producer.load_false_choices():
                records.append({
                    "domain_hash": to_ulonglong(choice.domain_hash),
                    "shingle": to_ulonglong(choice.shingle),
                    "shingle_type": choice.shingle_type,
                    "date": int(choice.date.timestamp())
                })
            yt.create_table(self.falses_table, attributes={"schema": schema})
            yt.write_table(self.falses_table, records, raw=False)

        self.comp_reputation_url = comp_reputation_host + '/api/v1'

    def test(self, chat: aiotg.Chat, match):
        return chat.send_text("[like this](https://sourceforge.net/p/telegram/wiki/markdown_syntax/)",
                              parse_mode='Markdown')

    async def get_reports_count(self, chat: aiotg.Chat, match):
        return await chat.send_text(f"{len(self.reports_groups)}")

    async def set_weight(self, chat: aiotg.Chat, match):
        text: str = chat.message["text"]
        space_index = text.rfind(' ')
        if space_index == -1:
            return await chat.send_text(f"weight={self.engine.get_config()[0]}")
        else:
            raw = text[space_index:]
            try:
                self.engine.update_config(weight=float(raw))
                return await chat.send_text(f"weight={self.engine.get_config()[0]}")
            except Exception as e:
                return await chat.send_text(f"cannot parse weight from {raw}: {e}")

    async def set_duration(self, chat: aiotg.Chat, match):
        text: str = chat.message["text"]
        space_index = text.rfind(' ')
        if space_index == -1:
            return chat.send_text(f"duration={self.engine.get_config()[1]}")
        else:
            raw = text[space_index:]
            try:
                self.engine.update_config(duration=int(raw))
                return chat.send_text(f"duration={self.engine.get_config()[1]}")
            except Exception as e:
                return chat.send_text(f"cannot parse weight from {raw}: {e}")

    def next_group_report(self) -> typing.Optional[ReportsSignatureGroup]:
        logger.info("next_group_report")
        try:
            if not self.reports_iterator:
                raise StopIteration()
            return next(self.reports_iterator)
        except StopIteration:
            if self.reports_groups:
                self.reports_iterator = iter(self.reports_groups.values())
                return next(self.reports_iterator)
            self.reports_groups = self.reports_producer.load_reports()
            if self.reports_groups:
                self.reports_iterator = iter(self.reports_groups.values())
                logger.info(f"self.reports_iterator: {self.reports_iterator}")
                return next(self.reports_iterator, None)
            return None

    async def send_current_group(self, chat: aiotg.Chat, match):
        if self.current_group is not None:
            return await self.send_view()
        else:
            return await chat.send_text("there is no reports yet")

    async def broadcast_group(self):
        while True:
            logger.info("broadcasting group")
            if self.current_group is not None:
                logger.info(f"broadcasting group:{self.current_group}")
                return await self.send_view()
            else:
                self.current_group = self.next_group_report()
                logger.info("broadcasting group sleeping")
                await asyncio.sleep(self.refresh_time)

    async def usage(self, chat: aiotg.Chat, match):
        text = "/d [int] get current duration if number is absent, set it otherwise\n" \
               "/w [float] get current weight if number is absent, set it otherwise\n" \
               "/c show current task\n" \
               "/l show tasks count"
        return await chat.send_text(text)

    async def start(self, chat: aiotg.Chat, match):

        username = chat.sender["username"]

        if not await self.user_validator.valid(username):
            logger.info(f"{username} tried connect to")
            return await chat.reply('Sorry, dude')

        session = self.engine.session()
        session.add(Chat(chat_id=chat.id, inviter=username))
        session.commit()

        return await chat.send_text(f'Hello {username}!')

    async def get_chats(self, chat: aiotg.Chat, match):
        session = self.engine.session()
        chats = [str(chat) for chat in session.query(Chat).all()]

        message = '\n'.join(chats) if chats else 'there is no chats yet'

        return await chat.send_text(message)

    def write_false(self, sig: Signature):
        try:
            records = [{
                "domain_hash": to_ulonglong(sig.domain_hash),
                "shingle": to_ulonglong(sig.shingle),
                "shingle_type": sig.shingle_type,
                "date": int(datetime.now().timestamp())
            }]

            yt.write_table(self.falses_table, records, raw=False)
        except Exception as e:
            logger.exception(e)

    async def just_iter(self, chat, message_id, button_uuid):
        self.current_group = self.next_group_report()
        asyncio.create_task(self.broadcast_group())
        Button.remove(button_uuid)
        await chat.edit_reply_markup(message_id, {})

    async def delete_current_and_iter(self, chat, message_id, button_uuid):
        del self.reports_producer[self.current_group.signature]
        del self.reports_groups[self.current_group.signature]
        self.reports_iterator = iter(self.reports_groups.values())
        await self.just_iter(chat, message_id, button_uuid)

    async def proceed_callback(self, chat: aiotg.Chat, cq: aiotg.CallbackQuery, match):
        button_uuid = match.group(1)
        button = Button.from_string(button_uuid)
        if button is None:
            return chat.reply("task outdated")
        username = cq.src['from']['username']
        message_id = chat.message["message_id"]

        await chat.reply(f"{username} selected option: {button.text}")

        if button == FalseButton:
            self.engine.add_choice(self.current_group.signature, Choice.FALSE, username)
            self.write_false(self.current_group.signature)
            await self.delete_current_and_iter(chat, message_id, button_uuid)
            return

        if button == LaterButton:
            await self.just_iter(chat, message_id, button_uuid)
            return

        try:
            await button.ban.ban(self.comp_reputation_url, 5)
            self.engine.add_choice(self.current_group.signature, Choice.BAN, username)
            await self.delete_current_and_iter(chat, message_id, button_uuid)
        except Exception as e:
            return chat.reply(f"request error {str(e)}, {traceback.format_exc()}")

    async def error(self, chat: aiotg.Chat, match):
        logger.error(f"error callback:{chat}, {match}")

    async def send_view(self):
        weight, duration = self.engine.get_config()
        choices_history = self.engine.get_choices_history(self.current_group.signature)

        session = self.engine.session()
        stids = []
        try:
            stids += await self.current_group.get_stids()

            for queue_id, uid, stid in stids:
                if not stid:
                    continue
                url = ReportsSignatureGroup.get_url_by_stid(stid)
                path = str(stid) + ".png"
                try:
                    image = await self.screenshooter.save_screenshot(url, path)
                except selenium.common.exceptions.TimeoutException as e:
                    for chat in session.query(Chat).all():
                        await self.bot.send_message(chat_id=chat.chat_id, text="timeout rendering page")
                        logger.exception(e)
                    continue

                for chat in session.query(Chat).all():
                    with open(image, mode='rb') as f:
                        await self.bot.api_call(
                            "sendDocument", chat_id=str(chat.chat_id), document=f, caption=f"[{queue_id},{uid}]({url})"
                        )
                os.remove(path)
                break

        except Exception as e:
            logger.exception(e)

        view: View = await self.current_group.to_view(weight, duration, choices_history, stids)

        logger.info(f"send_view:View:{View}")
        buttons = [button.to_markup() for button in view.buttons]
        markup = ujson.dumps({
            "type": "InlineKeyboardMarkup",
            "inline_keyboard": [
                buttons
            ],
        })
        logger.info(f"send_view text:{view.markup};markup:{markup}")

        session = self.engine.session()
        logger.info(f"{session.query(Chat).all()}")
        for chat in session.query(Chat).all():
            try:
                await self.bot.send_message(chat_id=chat.chat_id,
                                            text=view.markup,
                                            reply_markup=markup,
                                            parse_mode='Markdown'
                                            )
            except aiotg.BotApiError as e:
                logger.exception(view.markup, e)
                try:
                    await self.bot.send_message(chat_id=chat.chat_id,
                                                text=view.markup,
                                                reply_markup=markup
                                                )
                except Exception as e:
                    logger.exception(view.markup, e)
