import datetime
import enum
import logging
import typing

import ujson
from sqlalchemy import Column, Integer, String, Float, create_engine, BigInteger, Boolean, ForeignKey, DateTime, Enum, \
    and_
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker, Session, relationship, scoped_session

from mail.so.libs.python.cast import to_longlong, to_ulonglong
from .mail_id import MailId

_base = declarative_base()

logger = logging.Logger("models")


class Engine:
    def __init__(self, path_to_db: str):
        self._engine = create_engine(path_to_db, echo=False)
        self._session_maker = sessionmaker(bind=self._engine)
        self._session = scoped_session(self._session_maker)
        _base.metadata.create_all(self._engine)

        try:
            session = self.session()
            config = session.query(Config).get(42)
            if config is None:
                session.add(Config(id=42))
                session.commit()
        except Exception as e:
            logger.exception(e)

    def session(self) -> Session:
        return self._session()

    def get_config(self):
        session = self.session()
        config = session.query(Config).get(42)
        return config.weight, config.duration

    def update_config(self, weight=None, duration=None):
        session = self.session()
        config = session.query(Config).get(42)
        if weight is not None:
            config.weight = weight
        if duration is not None:
            config.duration = duration
        session.commit()

    def add_choice(self, signature: "Signature", choice: "Choice", login: str):
        session = self.session()
        config = session.query(Config).get(42)
        session.add(ChoicesHistory(
            shingle_type=signature.shingle_type,
            shingle=signature.shingle,
            domain_hash=signature.domain_hash,
            choice=choice,
            login=login,
            weight=config.weight,
            duration=config.duration,
        ))
        session.commit()

    def get_choices_history(self, signature: "Signature") -> typing.List["ChoicesHistory"]:
        session = self.session()
        return session \
            .query(ChoicesHistory) \
            .filter(and_(and_(ChoicesHistory.shingle_type == signature.shingle_type,
                              ChoicesHistory.shingle == signature.shingle),
                         ChoicesHistory.domain_hash == signature.domain_hash, )) \
            .order_by(ChoicesHistory.date.desc()) \
            .limit(5) \
            .all()


class Chat(_base):
    __tablename__ = 'chats'
    chat_id = Column(Integer, primary_key=True)
    inviter = Column(String(200))

    def __repr__(self):
        return f"id:{self.chat_id}, inviter:{self.inviter}"


class Config(_base):
    __tablename__ = 'config'
    id = Column(BigInteger, primary_key=True)
    weight = Column(Float, default=5.0)
    duration = Column(Integer, default=5)


class Table(_base):
    __tablename__ = 'tables'
    table = Column(String, primary_key=True)
    proceed = Column(Boolean, default=False)
    reports = relationship("Report")

    def __str__(self):
        return f"{self.table} {self.proceed}, reports: {len(self.reports)}"


class Signature:
    def __init__(self, shingle_type, shingle, domain_hash):
        self.shingle_type = shingle_type
        self.shingle = shingle
        self.domain_hash = domain_hash

    @property
    def full_shingle(self):
        return hex(self.__hash__())

    def __hash__(self):
        return to_ulonglong(self.shingle) ^ to_ulonglong(self.domain_hash)

    def __eq__(self, other: "Signature"):
        return self.shingle_type == other.shingle_type and self.shingle == other.shingle and self.domain_hash == other.domain_hash

    def __ne__(self, other: "Signature"):
        return not self.__eq__(other)


class Report(_base):
    __tablename__ = 'reports'
    sig_id = Column(BigInteger, primary_key=True)
    table = Column(String, ForeignKey('tables.table'), primary_key=True)
    shingle_type = Column(Integer)
    shingle = Column(BigInteger)
    domain_hash = Column(BigInteger)
    spams = Column(Integer, default=0)
    total = Column(Integer, default=0)
    queue_id = Column(String, nullable=True, default=None)
    uid = Column(String, nullable=True, default=None)
    message_id = Column(String, nullable=True, default=None)
    proceed = Column(Boolean, default=False)
    timeline = Column(String)

    @staticmethod
    def parse_signature(lines: typing.List[str]) -> typing.Optional["Signature"]:
        type_and_shingle = ujson.loads(lines[0].replace('\'', '"'))
        if len(type_and_shingle) < 2:
            return None
        return Signature(int(type_and_shingle[0], 10), int(type_and_shingle[1], 16), int(lines[1], 10))

    @staticmethod
    def parse(table, record, mail_id: MailId) -> typing.Optional["Report"]:

        signature = Report.parse_signature(record["signature"])
        if signature is None:
            return None

        report = Report(
            table=table,
            sig_id=to_longlong(int(record['sig_id'])),
            shingle_type=to_longlong(signature.shingle_type),
            shingle=to_longlong(signature.shingle),
            domain_hash=to_longlong(signature.domain_hash),
            spams=int(record["spam"]),
            total=int(record["total"]),
            timeline=record["timeline"],
        )

        if mail_id is not None:
            report.queue_id = mail_id.queue_id
            report.uid = mail_id.uid
            report.message_id = mail_id.message_id

        return report

    def __str__(self):
        return f"({self.sig_id},{self.shingle},{self.shingle_type})"

    @property
    def id(self):
        return f"{self.table}_{to_ulonglong(self.sig_id)}"

    @property
    def signature(self):
        return Signature(self.shingle_type, self.shingle, self.domain_hash)


class Choice(enum.Enum):
    BAN = 1,
    FALSE = 2,


class ChoicesHistory(_base):
    __tablename__ = 'choices'
    shingle_type = Column(Integer, primary_key=True)
    shingle = Column(BigInteger, primary_key=True)
    domain_hash = Column(BigInteger, primary_key=True)
    date = Column(DateTime, default=datetime.datetime.now, primary_key=True)
    choice = Column(Enum(Choice))
    login = Column(String)
    weight = Column(Float, default=30)
    duration = Column(Integer, default=240)

    def __str__(self):
        return f"{self.date}: {self.choice} by {self.login}, {self.duration} {self.weight}"

    def __repr__(self):
        return self.__str__()
