import logging
import typing
from itertools import chain

import ujson
from aiohttp import ClientSession

from .ban import Ban
from .message_builder import Builder
from .models import Report, Signature, ChoicesHistory
from .timeline import Timeline
from .view import LaterButton, FalseButton, View, Button

logger = logging.Logger("reports_group")


class ReportsSignatureGroup:
    def __init__(self, signature: Signature):
        self.signature = signature
        self.reports: typing.List[Report] = []

    def add(self, report: Report):
        assert self.signature == report.signature
        self.reports.append(report)

    def merge(self, other: "ReportsSignatureGroup"):
        assert self.signature == other.signature
        self.reports.extend(other.reports)

    @staticmethod
    def get_url_by_stid(stid):
        return f'http://so-web.n.yandex-team.ru/stid_show.pl?stid={stid}'

    async def get_stids(self):
        stids = set()
        async with ClientSession(conn_timeout=1) as session:
            for report in self.reports:
                logger.info(report)
                if not report.queue_id or not report.uid:
                    continue
                async with session.get(f'http://new-msearch-proxy.mail.yandex.net:8051/sequential/search?'
                                       f'service=change_log&'
                                       f'prefix={report.uid}&'
                                       f'text=smtp_id:{report.queue_id}&'
                                       f'get=stid') as resp:
                    # {"hitsCount": 1, "hitsArray": [{"stid": "320.mail:0.E3309825:29305448216592549222270175953"}]}
                    if resp.content_type != 'application/json':
                        stids.add((report.queue_id, report.uid, None))
                        continue
                    data = await resp.json()
                    if resp.status != 200:
                        logger.error(f"{resp.status}, {data}")
                        continue
                    hits_array = data.get("hitsArray")
                    if len(hits_array) == 0:
                        continue
                    stid = hits_array[0].get("stid")
                    stids.add((report.queue_id, report.uid, stid))

        return list(stids)

    async def to_view(self, weight: float, duration: int, choices_history: typing.List[ChoicesHistory],
                      stids: typing.Optional[list]) -> View:
        row_timelines = chain(*(ujson.loads(report.timeline) for report in self.reports))
        sorted_timelines = sorted(set(Timeline(v) for v in row_timelines), reverse=True)

        builder = Builder(4000)

        builder += "*stids*:\n"

        if not stids:
            try:
                stids = await self.get_stids()
            except Exception as e:
                builder += str(e)

        if stids:
            for queue_id, uid, stid in stids[:5]:
                builder += f"[{queue_id},{uid}]({self.get_url_by_stid(stid)})\n"

        builder += "*history*:\n"
        for choice in choices_history:
            builder += str(choice) + '\n'

        builder += "*subj/from*:\n"
        for tl in sorted_timelines:
            builder += str(tl) + '\n'

        ban = Ban(self.signature.shingle_type,
                  hash(self.signature),
                  duration=duration, weight=weight)

        return View(builder.build(), buttons=[
            Button(text=f"ban {self.signature.shingle_type}:{hex(hash(self.signature))}", ban=ban),
            FalseButton, LaterButton
        ])
