package ru.yandex.mts_api_proxy;

import java.util.concurrent.Future;

import javax.annotation.Nonnull;

import org.apache.http.Header;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.util.CallbackFutureBase;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonBadCastException;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.mts_api_proxy.config.ImmutableMtsApiClientConfig;

public class MtsApiClient extends AbstractAsyncClient<MtsApiClient> {
    @Nonnull
    private final ImmutableMtsApiClientConfig config;

    public MtsApiClient(
            @Nonnull SharedConnectingIOReactor reactor,
            @Nonnull ImmutableMtsApiClientConfig config) {
        super(reactor, config);
        this.config = config;
    }

    private MtsApiClient(
            @Nonnull final CloseableHttpAsyncClient client,
            @Nonnull final MtsApiClient sample) {
        super(client, sample);
        this.config = sample.config;
    }

    @Override
    protected MtsApiClient adjust(CloseableHttpAsyncClient client) {
        return new MtsApiClient(client, this);
    }

    public void score(long msisdn, String token, @Nonnull FutureCallback<Double> callback) {
        final BasicAsyncRequestProducerGenerator requestProducerGenerator =
                new BasicAsyncRequestProducerGenerator(
                        "/",
                        "{\"msisdn\":" + msisdn + ",\"agreement\":true,\"features\":[{\"name\":\"frod_score\"}]}",
                        ContentType.APPLICATION_JSON);

        final Header bearerAuthorizationHeader = new BasicHeader(
                "Authorization",
                "Bearer " + token);

        requestProducerGenerator.addHeader(bearerAuthorizationHeader);
        requestProducerGenerator.addHeader(config.destinationScoreUrlHeader());
        requestProducerGenerator.addHeader(config.clientIdHeader());

        execute(
                config.host(),
                requestProducerGenerator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                new ScoreParsingCallback(callback));
    }

    public Future<String> token(@Nonnull FutureCallback<String> callback) {
        final BasicAsyncRequestProducerGenerator requestProducerGenerator =
                new BasicAsyncRequestProducerGenerator(
                        "/",
                        "grant_type=client_credentials",
                        ContentType.APPLICATION_FORM_URLENCODED);

        requestProducerGenerator.addHeader(config.basicAuthorizationHeader());
        requestProducerGenerator.addHeader(config.destinationTokenUrlHeader());
        requestProducerGenerator.addHeader(config.clientIdHeader());

        final TokenParsingCallback tokenParsingCallback = new TokenParsingCallback(callback);
        execute(
                config.host(),
                requestProducerGenerator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                tokenParsingCallback);
        return tokenParsingCallback;
    }

    private static class TokenParsingCallback
            extends CallbackFutureBase<String, JsonObject> {

        protected TokenParsingCallback(FutureCallback<? super String> callback) {
            super(callback);
        }

        @Override
        protected String convertResult(JsonObject result) {
            try {
                return result.get("access_token").asString();
            } catch (JsonBadCastException e) {
                onFailure(e);
                return null;
            }
        }
    }

    private static class ScoreParsingCallback
            extends CallbackFutureBase<Double, JsonObject> {

        protected ScoreParsingCallback(FutureCallback<? super Double> callback) {
            super(callback);
        }

        @Override
        protected Double convertResult(JsonObject result) {
            try {
                return result.get("features").asList().get(0).get("result").asDouble();
            } catch (JsonBadCastException e) {
                onFailure(e);
                return 0.;
            }
        }
    }
}
